/**++
 *   
 *   LICENSE
 *   -------
 *   
 *   Copyright (c) 2004 Renato Mancuso
 *   All rights reserved.
 *   
 *   Redistribution and use in source and binary forms, with or without modification, are 
 *   permitted provided that the following conditions are met:
 *   
 *   - Redistributions of source code must retain the above copyright notice, this list 
 *     of conditions and the following disclaimer.
 *   
 *   - Redistributions in binary form must reproduce the above copyright notice, this list
 *     of conditions and the following disclaimer in the documentation and/or other materials 
 *     provided with the distribution.
 *   
 *   - Neither the name of Renato Mancuso nor the names of its contributors may be used to 
 *     endorse or promote products derived from this software without specific prior written 
 *     permission.
 *   
 *   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS AS IS AND ANY EXPRESS 
 *   OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY 
 *   AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR 
 *   CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL 
 *   DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, 
 *   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER 
 *   IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT 
 *   OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *   
--**/


#ifndef FF2XML_XML_WRITER_INCLUDED
#define FF2XML_XML_WRITER_INCLUDED

#if !defined(OPEN_EMBL_NO_PRAGMA_ONCE)
#pragma once
#endif

#include <cstdio>
#include <vector>
#include <string>

#include "XmlAttribute.hpp"
#include "NonCopyable.hpp"

namespace OpenEMBL
{
namespace ff2xml
{
    

    typedef std::vector<XmlAttribute> XmlAttributeList;
    typedef std::vector<std::string>  StringList;


    class XmlWriter
        : OpenEMBL::Phoenix::NonCopyable
    {
    public:
        //
        // embedded types
        //
        enum layout_type { SINGLELINE, MULTILINE };

        //
        // construction
        //
        explicit XmlWriter(const char* destFilename);
        XmlWriter();
        ~XmlWriter();

        //
        // open/close connection to external file 
        //
        void open(const char* destFilename);
        void close();

        //
        // direct write (untranslated)
        //
        void directWrite(char c);
        void directWrite(char const * str);
        void directWrite(std::string const & str) { directWrite(str.c_str()); }

        //
        // primitives
        //
        void newLine();
        void writeIndent(unsigned level);
        void openTag();
        void closeTag(bool single = false);
        void writeElementName(std::string const & str);
        void writeElementName(char const * str);
        void writeElementText(std::string const & str);
        void writeElementText(char const* str);        
        void writeAttribute(XmlAttribute const & attr);        

        //
        // composite operation
        //
        void writeXMLDeclaration(char const * version, char const * encoding);

        void writeEndTag(char const * tagName);

        void writeBeginTag
            (
            unsigned                    indentLevel,
            char const *                tagName
            );

        void writeBeginTag
            (
            layout_type                 layout,
            unsigned                    indentLevel,
            char const *                tagName, 
            XmlAttributeList const &    attributes,
            bool                        isSingleElement = false
            ); 

        void writeSingleTag(
            layout_type                 layout,
            unsigned                    indentLevel,
            char const *                tagName,
            XmlAttributeList const &    attributes
            );

        void writeElement 
            (
            unsigned                    indentLevel,
            char const *                tagName,
            XmlAttributeList            attributes,
            std::string const &         text
            );

        void writeElement  
            (
            unsigned                    indentLevel,
            char const *                tagName,
            XmlAttributeList            attributes,
            StringList const &          text
            );

    private:
        void raiseSysError(char const * msg) const;
        void writeAttributeValue(char const * str);
        void writeAttributeValue(std::string const & str);
        void writeBeginTagSingleLine 
            (
            unsigned                    indentLevel,
            char const *                tagName, 
            XmlAttributeList const &    attributes,
            bool                        isSingleElement
            );
        void writeBeginTagMultiLine
            (
            unsigned                    indentLevel,
            char const *                tagName,         
            XmlAttributeList const &    attributes,
            bool                        isSingleElement /* = false */
            );      

    private:
        FILE* _destinationFile;
    };

}
}

#endif // FF2XML_XML_WRITER_INCLUDED
