/**++
 *   
 *   LICENSE
 *   -------
 *   
 *   Copyright (c) 2004 Renato Mancuso
 *   All rights reserved.
 *   
 *   Redistribution and use in source and binary forms, with or without modification, are 
 *   permitted provided that the following conditions are met:
 *   
 *   - Redistributions of source code must retain the above copyright notice, this list 
 *     of conditions and the following disclaimer.
 *   
 *   - Redistributions in binary form must reproduce the above copyright notice, this list
 *     of conditions and the following disclaimer in the documentation and/or other materials 
 *     provided with the distribution.
 *   
 *   - Neither the name of Renato Mancuso nor the names of its contributors may be used to 
 *     endorse or promote products derived from this software without specific prior written 
 *     permission.
 *   
 *   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS AS IS AND ANY EXPRESS 
 *   OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY 
 *   AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR 
 *   CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL 
 *   DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, 
 *   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER 
 *   IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT 
 *   OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *   
--**/


#include "ACStarLineParser.hpp"
#include "ParserErrors.hpp"
#include "ParserWarnings.hpp"
#include "StringUtils.hpp"
#include "Regex.hpp"

#include <vector>
#include <string>
#include <cstdlib>
#include <cassert>

namespace OpenEMBL
{
namespace Phoenix
{

    ACStarLineParser::ACStarLineParser(
        IParserCtx*                 pParserCtx,
        IItemHandler<ACStarLine>*   pItemHandler)
        : ItemParserImpl<ACStarLine>(pParserCtx, pItemHandler)
    {
    }

    METHODIMP ACStarLineParser::parse()
    {
        long theLineNumber = currentLineNumber();        

        if (NULL == currentLine())
        {
            logError(theLineNumber,
                     ERROR_EOF_FOUND,
                     NULL);
            return;
        }

        if (!isACStarLine(currentLine()))
        {
            logError(theLineNumber,
                     ERROR_INVALID_AC_STAR_LINE,
                     currentLine());

            return;
        }

        std::string theLine = currentLine();
        nextLine();

        processLine(getLineContent(theLine), theLineNumber);        
    }

    bool ACStarLineParser::isACStarLine(char const * theLine)
    {
        return 0 == strncmp(theLine, "AC * ", 5);
    }

    std::string ACStarLineParser::getLineContent(std::string const & theLine)
    {
        return trim(theLine.substr(5));
    }

    void ACStarLineParser::processLine(const std::string & theLine, long theLineNumber)
    {
        //
        //  AC* line
        //
        // Description:
        //    The AC* ine is an internal line type used by genome
        //    project submittors to supply their internal identifiers. 
        //
        // Line format:
        //
        //    AC * _GP_ID[.GP_VERSION] [WGS_VERSION]
        //
        //    where:
        //    - _GP_ID:      alphanumeric string starting with an underscore
        //    - GP_VERSION:  integer > 0  
        //    - WGS_VERSION: integer 1-99 (only for WGS projects!)
        //

        //
        // Regex is: ^(_[^\s\.]{0,})(?:\.(\d+)){0,1}(?:\s+(\d{1,2})){0,1}$
        //

        static Regex theRegex("^(_[^\\s\\.]{0,})(?:\\.(\\d+)){0,1}(?:\\s+(\\d{1,2})){0,1}$");
        Match       theMatches;

        if (!regexSearch(theLine, theMatches, theRegex))
        {
            logError(theLineNumber,
                     ERROR_INVALID_AC_STAR_LINE,
                     theLine.c_str());

            return;
        }

        ACStarLine theField;

        theField.GenomeProjectID = theMatches.str(1);

        if (theMatches[2].matched)
            theField.Version = atoi(theMatches.str(2).c_str());

        if (theMatches[3].matched)
            theField.WGSVersion = atoi(theMatches.str(3).c_str());

        notifyParsed(theLineNumber, theField);
    }

}
}
