/**++
 *   
 *   LICENSE
 *   -------
 *   
 *   Copyright (c) 2004 Renato Mancuso
 *   All rights reserved.
 *   
 *   Redistribution and use in source and binary forms, with or without modification, are 
 *   permitted provided that the following conditions are met:
 *   
 *   - Redistributions of source code must retain the above copyright notice, this list 
 *     of conditions and the following disclaimer.
 *   
 *   - Redistributions in binary form must reproduce the above copyright notice, this list
 *     of conditions and the following disclaimer in the documentation and/or other materials 
 *     provided with the distribution.
 *   
 *   - Neither the name of Renato Mancuso nor the names of its contributors may be used to 
 *     endorse or promote products derived from this software without specific prior written 
 *     permission.
 *   
 *   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS AS IS AND ANY EXPRESS 
 *   OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY 
 *   AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR 
 *   CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL 
 *   DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, 
 *   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER 
 *   IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT 
 *   OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *   
--**/


#include "COLineLocationParser.hpp"
#include "ParserErrors.hpp"
#include "ParserWarnings.hpp"

#include <cstdio>
#include <sstream>
#include <cassert>

namespace OpenEMBL
{
namespace Phoenix
{

    COLineLocationParser::COLineLocationParser(
        IParserCtx*             theParserCtx,
        IContigInfoHandler*     theHandler)
        : m_ParserCtx(theParserCtx)
        , m_Handler(theHandler)
    {
        m_Lexer.setErrorLog(this);
        m_Lexer.setTextSource(this);
    }

    void COLineLocationParser::notifyBeginContigInfo(long theLineNumber, bool isComplement)
    {
        if (m_Handler)
            m_Handler->onBeginContigInfo(theLineNumber, isComplement);
    }

    void COLineLocationParser::notifyParsed(long theLineNumber, COSegmentInfo const & theItem)
    {
        if (m_Handler)
            m_Handler->onParsed(theLineNumber, theItem);
    }

    void COLineLocationParser::notifyParsed(long theLineNumber, COGapInfo const & theItem)
    {
        if (m_Handler)
            m_Handler->onParsed(theLineNumber, theItem);
    }

    void COLineLocationParser::notifyEndContigInfo()
    {
        if (m_Handler)
            m_Handler->onEndContigInfo();
    }


    METHODIMP_(LPCSTR) 
    COLineLocationParser::currentLine() const NO_THROW
    {
        if (NULL != m_ParserCtx)
            return m_ParserCtx->currentLine();

        return NULL;
    }

    METHODIMP_(long)   
    COLineLocationParser::currentLineNumber() const NO_THROW
    {
        if (NULL != m_ParserCtx)
            return m_ParserCtx->currentLineNumber();

        return TEXTSOURCE_INVALID_LINE_NUMBER;
    }

    METHODIMP_(bool)   
    COLineLocationParser::nextLine()
    {
        if (NULL != m_ParserCtx)
            return m_ParserCtx->nextLine();

        return false;
    }

    METHODIMP 
    COLineLocationParser::logError(long theLineNumber, int theErrorCode, LPCSTR theMessage)
    {
        if (NULL != m_ParserCtx)
            return m_ParserCtx->logError(theLineNumber, theErrorCode, theMessage);
    }

    METHODIMP 
    COLineLocationParser::logWarning(long theLineNumber, int theWarning, LPCSTR theMessage)
    {
        if (NULL != m_ParserCtx)
            return m_ParserCtx->logWarning(theLineNumber, theWarning, theMessage);
    }

    std::string COLineLocationParser::toString(int theToken)
    {
        switch (theToken)
        {
        case EOF:
            return "<END OF LOCATION>";

        case BAD_TOKEN:
            return "<BAD_TOKEN>";

        case ACCESSION_NUMBER:
            return "accession number";

        case NUMBER:
            return "number";

        case DOUBLE_DOT:
            return "..";

        case JOIN:
            return "join";

        case ORDER:
            return "order";

        case COMPLEMENT:
            return "complement";

        case GAP:
            return "gap";

        case COMMA:
            return ",";

        case DOT:
            return ".";

        case COLON:
            return ":";

        case CARET:
            return "^";

        case OPENING_PARENTHESIS:
            return "(";

        case CLOSING_PARENTHESIS:
            return ")";

        case OPENING_ANGLE_BRACKET:
            return "<";

        case CLOSING_ANGLE_BRACKET:
            return ">";

        default:
            assert(0);
        }

        return "<UNKNOWN TOKEN>";
    }

    void COLineLocationParser::parse()
    {
        try
        {
            parseLocation();
        }
        catch (InvalidLocation &)
        {         
            /* DO NOTHING */;
        }
    }

    int COLineLocationParser::getToken()
    {
        int theToken = m_Lexer.getToken();

        switch (theToken)
        {
        case EOF:
            logError(currentLineNumber(),
                    ERROR_UNEXPECTED_LOCATION_END,
                    NULL);

            throw InvalidLocation();

        case BAD_TOKEN:
            // message should already have been provided by Lexer
            throw InvalidLocation();

        case OPENING_ANGLE_BRACKET:
        case CLOSING_ANGLE_BRACKET:
        case ORDER:
        case CARET: // these tokens are illegal inside a CO line location
            logError(currentLineNumber(),
                    ERROR_BAD_TOKEN_IN_LOCATION,
                    NULL);
            throw InvalidLocation();
        }

        return theToken;
    }

    int COLineLocationParser::peekToken() const throw()
    {
        return m_Lexer.peekToken();
    }

    std::string COLineLocationParser::getTokenValue() const
    {
        std::string theValue;
        
        if (!m_Lexer.getTokenValue(theValue))
        {
            assert(0);
        }           

        return theValue;
    }

    void COLineLocationParser::badToken(int theExpectedToken, int theFoundToken)
    {
        std::ostringstream oss;

        oss << "expected '" << toString(theExpectedToken)
            << "' found '"  << getTokenValue()            
            << "' [token type: '" << toString(theFoundToken)
            << "']"
            ;

        logError(currentLineNumber(),
                 ERROR_BAD_TOKEN_IN_LOCATION,
                 oss.str().c_str());

        throw InvalidLocation();
    }

    void COLineLocationParser::readToken(int whichToken)
    {
        int theToken = getToken();

        if (theToken != whichToken)
            badToken(whichToken, theToken);
    }

    void COLineLocationParser::readToken(int whichToken, std::string & theValue)
    {
        readToken(whichToken);       
        theValue = getTokenValue();
    }

    void COLineLocationParser::readAccessionNumber(std::string & theValue)
    {
        readToken(ACCESSION_NUMBER, theValue);
    }

    void COLineLocationParser::readNumber(std::string & theValue)
    {
        readToken(NUMBER, theValue);
    }

    void COLineLocationParser::readNumber(long & theValue)
    {
        std::string theRep;
        readNumber(theRep);

        theValue = atol(theRep.c_str());
    }

    void COLineLocationParser::readDoubleDot()
    {
        readToken(DOUBLE_DOT);
    }

    void COLineLocationParser::readJoin()
    {
        readToken(JOIN);
    }

    void COLineLocationParser::readComplement()
    {
        readToken(COMPLEMENT);
    }

    void COLineLocationParser::readComma()
    {
        readToken(COMMA);
    }

    void COLineLocationParser::readDot()
    {
        readToken(DOT);
    }

    void COLineLocationParser::readColon()
    {
        readToken(COLON);
    }

    void COLineLocationParser::readOpeningParenthesis()
    {
        readToken(OPENING_PARENTHESIS);
    }

    void COLineLocationParser::readClosingParenthesis()
    {
        readToken(CLOSING_PARENTHESIS);
    }

    void COLineLocationParser::readGap()
    {
        readToken(GAP);
    }

    void COLineLocationParser::parseLocation()
    {
        bool isComplement = false;

        if (peekToken() == COMPLEMENT)
        {
            isComplement = true;
            readComplement();
            readOpeningParenthesis();
        }

        notifyBeginContigInfo(currentLineNumber(), isComplement);

        readJoin();
        readOpeningParenthesis();
        readElement();
        readComma();
            
        while (1)
        {
            readElement();

            if (peekToken() == CLOSING_PARENTHESIS)
                break;

            readComma();
        }

        readClosingParenthesis();

        if (isComplement)
            readClosingParenthesis();

        if (peekToken() != EOF)
            badToken(EOF, getToken());

        notifyEndContigInfo();
    }

    void COLineLocationParser::readElement()
    {       
        if (peekToken() == GAP)
            readGapElement();
        else
            readSpanElement();
    }

    void COLineLocationParser::readGapElement()
    {
        COGapInfo theItem;
        long      theLineNumber = currentLineNumber();
        
        readGap();
        readOpeningParenthesis();

        if (peekToken() == NUMBER)
            readNumber(theItem.Length);

        readClosingParenthesis();
        notifyParsed(theLineNumber, theItem);
    }

    void COLineLocationParser::readSpanElement()
    {
        COSegmentInfo   theItem;
        long            theLineNumber = currentLineNumber();

        bool isComplement = false;

        if (peekToken() == COMPLEMENT)
        {
            isComplement = true;
            readComplement();
            readOpeningParenthesis();
            theItem.Complement = true;
        }

        readAccessionNumber(theItem.AccessionNumber);
        readDot();
        readNumber(theItem.SequenceVersion);
        readColon();
        readNumber(theItem.Span.First);

        if (peekToken() == DOUBLE_DOT)
        {
            readDoubleDot();
            readNumber(theItem.Span.Last);
        }

        if (isComplement)
            readClosingParenthesis();
    
        notifyParsed(theLineNumber, theItem);
    }

}
}

