/**++
 *   
 *   LICENSE
 *   -------
 *   
 *   Copyright (c) 2004 Renato Mancuso
 *   All rights reserved.
 *   
 *   Redistribution and use in source and binary forms, with or without modification, are 
 *   permitted provided that the following conditions are met:
 *   
 *   - Redistributions of source code must retain the above copyright notice, this list 
 *     of conditions and the following disclaimer.
 *   
 *   - Redistributions in binary form must reproduce the above copyright notice, this list
 *     of conditions and the following disclaimer in the documentation and/or other materials 
 *     provided with the distribution.
 *   
 *   - Neither the name of Renato Mancuso nor the names of its contributors may be used to 
 *     endorse or promote products derived from this software without specific prior written 
 *     permission.
 *   
 *   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS AS IS AND ANY EXPRESS 
 *   OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY 
 *   AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR 
 *   CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL 
 *   DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, 
 *   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER 
 *   IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT 
 *   OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *   
--**/


#include "ContigInfoParser.hpp"
#include "ParserErrors.hpp"
#include "ParserWarnings.hpp"
#include "StringUtils.hpp"
#include "COLineTextSourceAdapter.hpp"
#include "ParserCtxOnErrorLogAdapter.hpp"
#include "COLineLocationParser.hpp"

#include <cstring>
#include <cassert>

namespace OpenEMBL
{
namespace Phoenix
{

    ContigInfoParser::ContigInfoParser(
        IParserCtx*             theParserCtx,
        IContigInfoHandler*     theHandler)
        : m_ParserCtx(theParserCtx)
        , m_Handler(theHandler)
    {
    }

    METHODIMP_(LPCSTR) ContigInfoParser::currentLine() const NO_THROW
    {
        if (NULL == m_ParserCtx)
            return NULL;

        return m_ParserCtx->currentLine();
    }

    METHODIMP_(long)   ContigInfoParser::currentLineNumber() const NO_THROW
    {
        if (NULL == m_ParserCtx)
            return TEXTSOURCE_INVALID_LINE_NUMBER;

        return m_ParserCtx->currentLineNumber();
    }

    METHODIMP_(bool)   ContigInfoParser::nextLine()
    {
        if (NULL == m_ParserCtx)
            return false;

        return m_ParserCtx->nextLine();
    }

    METHODIMP ContigInfoParser::logError(long theLineNumber, int theErrorCode, char const * theMessage)
    {
        if (m_ParserCtx)
            m_ParserCtx->logError(theLineNumber, theErrorCode, theMessage);
    }

    METHODIMP ContigInfoParser::logWarning(long theLineNumber, int theWarning, char const * theMessage)
    {
        if (m_ParserCtx)
            m_ParserCtx->logWarning(theLineNumber, theWarning, theMessage);
    }

    bool ContigInfoParser::isCOLine(char const * theLine)
    {
        return 0 == strncmp(theLine, "CO   ", 5);
    }

    METHODIMP ContigInfoParser::parse()
    {
        long    theLineNumber = currentLineNumber();
        LPCSTR  theLine       = currentLine();

        if (NULL == theLine)
        {
            logError(theLineNumber,
                     ERROR_EOF_FOUND,
                     NULL);

            return;
        }

        if (!isCOLine(theLine))
        {
            logError(theLineNumber,
                     ERROR_INVALID_CO_LINE,
                     theLine);
            nextLine();
            return;
        }

        processLines();

        // avoid endless loop
        if (currentLineNumber() == theLineNumber)
            nextLine();
    }

    void ContigInfoParser::processLines()
    {
        COLineTextSourceAdapter     theTextSourceAdapter(this);
        ParserCtxOnErrorLogAdapter  theInterceptionParserCtx(this, &theTextSourceAdapter);

        COLineLocationParser        theParser(&theInterceptionParserCtx, m_Handler);
        theParser.parse();
    }

}
}
