/**++
 *   
 *   LICENSE
 *   -------
 *   
 *   Copyright (c) 2004 Renato Mancuso
 *   All rights reserved.
 *   
 *   Redistribution and use in source and binary forms, with or without modification, are 
 *   permitted provided that the following conditions are met:
 *   
 *   - Redistributions of source code must retain the above copyright notice, this list 
 *     of conditions and the following disclaimer.
 *   
 *   - Redistributions in binary form must reproduce the above copyright notice, this list
 *     of conditions and the following disclaimer in the documentation and/or other materials 
 *     provided with the distribution.
 *   
 *   - Neither the name of Renato Mancuso nor the names of its contributors may be used to 
 *     endorse or promote products derived from this software without specific prior written 
 *     permission.
 *   
 *   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS AS IS AND ANY EXPRESS 
 *   OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY 
 *   AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR 
 *   CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL 
 *   DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, 
 *   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER 
 *   IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT 
 *   OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *   
--**/


#include "DRLineParser.hpp"
#include "ParserErrors.hpp"
#include "ParserWarnings.hpp"
#include "StringUtils.hpp"
#include "StringTokenizer.hpp"

#include <vector>
#include <string>
#include <cstdlib>
#include <cassert>

namespace OpenEMBL
{
namespace Phoenix
{

    DRLineParser::DRLineParser(
        IParserCtx*             pParserCtx,
        IItemHandler<DRLine>*   pItemHandler)
        : ItemParserImpl<DRLine>(pParserCtx, pItemHandler)
    {
    }

    METHODIMP DRLineParser::parse()
    {
        long theLineNumber = currentLineNumber();        

        if (NULL == currentLine())
        {
            logError(theLineNumber,
                     ERROR_EOF_FOUND,
                     NULL);
            return;
        }

        std::string theLine = currentLine();
        nextLine();

        processLine(theLine, theLineNumber);        
    }

    void DRLineParser::processLine(const std::string & theLine, long theLineNumber)
    {
        if (0 != strncmp(theLine.c_str(), "DR   ", 5))    
        {
            logError(theLineNumber,
                     ERROR_INVALID_DR_LINE,
                     theLine.c_str());
            return;
        }

        std::vector<std::string> theIdentifiers;

        StringTokenizer theTokenizer(" \t");
        theTokenizer.tokenize(theLine.substr(5), theIdentifiers);

        if (theIdentifiers.size() < 2 || theIdentifiers.size() > 3)
        {
            logError( theLineNumber,
                      ERROR_INVALID_DR_LINE,
                      theLine.c_str() );
            return;
        }

        DRLine theField;

        theField.DatabaseIdentifier = theIdentifiers[0];
        theField.PrimaryIdentifier  = theIdentifiers[1];

        if (theIdentifiers.size() == 3)
            theField.SecondaryIndentifier = theIdentifiers[2];

        removeTerminator(theField.DatabaseIdentifier, ';');
        removeTerminator(theField.PrimaryIdentifier, theIdentifiers.size() == 3 ? ';' : '.');
        removeTerminator(theField.SecondaryIndentifier, '.');        

        notifyParsed(theLineNumber, theField);
    }

}
}
