/**++
 *   
 *   LICENSE
 *   -------
 *   
 *   Copyright (c) 2004 Renato Mancuso
 *   All rights reserved.
 *   
 *   Redistribution and use in source and binary forms, with or without modification, are 
 *   permitted provided that the following conditions are met:
 *   
 *   - Redistributions of source code must retain the above copyright notice, this list 
 *     of conditions and the following disclaimer.
 *   
 *   - Redistributions in binary form must reproduce the above copyright notice, this list
 *     of conditions and the following disclaimer in the documentation and/or other materials 
 *     provided with the distribution.
 *   
 *   - Neither the name of Renato Mancuso nor the names of its contributors may be used to 
 *     endorse or promote products derived from this software without specific prior written 
 *     permission.
 *   
 *   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS AS IS AND ANY EXPRESS 
 *   OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY 
 *   AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR 
 *   CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL 
 *   DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, 
 *   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER 
 *   IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT 
 *   OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *   
--**/


#include "DTLineParser.hpp"
#include "ParserErrors.hpp"
#include "ParserWarnings.hpp"
#include "StringUtils.hpp"
#include "Regex.hpp"

#include <algorithm>
#include <cstdlib>
#include <cstring>
#include <cassert>

namespace OpenEMBL
{
namespace Phoenix
{

    DTLineParser::DTLineParser(
        IParserCtx*                 pParserCtx,
        IItemHandler<DTLine>*       pItemHandler)
        : ItemParserImpl<DTLine>(pParserCtx, pItemHandler)
    {
    }

    METHODIMP DTLineParser::parse()
    {
        long   theLineNumber = currentLineNumber();
        LPCSTR theLine       = currentLine();

        if (NULL == theLine)
        {
            logError(theLineNumber,
                     ERROR_EOF_FOUND,
                     NULL);
            return;
        }

        if (!isDTLine(theLine))
        {
            logError(theLineNumber,
                     ERROR_INVALID_DT_LINE,
                     theLine);
            nextLine();
            return;
        }

        StringList theLines;
        extractLines(isDTLine, theLines);
        processLines(theLines, theLineNumber);
    }

    bool DTLineParser::isDTLine(const char* theLine)
    {
        assert(NULL != theLine);
        return 0 == strncmp(theLine, "DT   ", 5);
    }

    String DTLineParser::getLineContent(const String & theLine)
    {
        assert(theLine.length() >= 5);
        return trimRight(theLine.substr(5));
    }

    void DTLineParser::processLines(const StringList & theLines, long theLineNumber)
    {
        if (theLines.size() != 2)
        {
            logError(theLineNumber,
                     ERROR_INVALID_DT_LINE,
                     "there must be exactly 2 DT lines");
            return;
        }

        DTLine theField;

        if (!parseCreated(theLineNumber, theLines[0], theField))
            return;

        if (!parseLastUpdated(theLineNumber + 1, theLines[1], theField))
            return;

        notifyParsed(theLineNumber, theField);
    }

    bool DTLineParser::parseCreated( 
        long                theLineNumber,
        std::string const & theLine, 
        DTLine            & theField )
    {
        //
        //  Regex is: ^DT\s+(\d{2}-[A-Z]{3}-\d{4})\s*\(Rel\.\s*(\d+), Created\)\s*$
        //
        static Regex theRegex( 
            "^DT\\s+(\\d{2}-[A-Z]{3}-\\d{4})\\s*\\(Rel\\.\\s*(\\d+), Created\\)\\s*$",
            PCRE_CASELESS );

        Match theMatches;

        if (!regexSearch(theLine, theMatches, theRegex))
        {
            logError(theLineNumber,
                     ERROR_INVALID_DT_LINE,
                     theLine.c_str());
            return false;
        }

        theField.FirstCreated   = theMatches.str(1);
        theField.ReleaseCreated = atoi(theMatches.str(2).c_str());

        return true;
    }


    bool DTLineParser::parseLastUpdated( 
        long                theLineNumber,
        std::string const & theLine, 
        DTLine            & theField )
    {
        //
        //  Regex is: ^DT\s+(\d{2}-[A-Z]{3}-\d{4})\s*\(Rel\.\s*(\d+), Last updated, Version (\d+)\)\s*$
        //
        static Regex theRegex( 
            "^DT\\s+(\\d{2}-[A-Z]{3}-\\d{4})\\s*\\(Rel\\.\\s*(\\d+), Last updated, Version (\\d+)\\)\\s*$",
            PCRE_CASELESS );

        Match theMatches;

        if (!regexSearch(theLine, theMatches, theRegex))
        {
            logError(theLineNumber,
                     ERROR_INVALID_DT_LINE,
                     theLine.c_str());
            return false;
        }

        theField.LastUpdated        = theMatches.str(1);
        theField.ReleaseLastUpdated = atoi(theMatches.str(2).c_str());
        theField.ExternalVersion    = atoi(theMatches.str(3).c_str());

        return true;
    }

}
}
