/**++
 *   
 *   LICENSE
 *   -------
 *   
 *   Copyright (c) 2004 Renato Mancuso
 *   All rights reserved.
 *   
 *   Redistribution and use in source and binary forms, with or without modification, are 
 *   permitted provided that the following conditions are met:
 *   
 *   - Redistributions of source code must retain the above copyright notice, this list 
 *     of conditions and the following disclaimer.
 *   
 *   - Redistributions in binary form must reproduce the above copyright notice, this list
 *     of conditions and the following disclaimer in the documentation and/or other materials 
 *     provided with the distribution.
 *   
 *   - Neither the name of Renato Mancuso nor the names of its contributors may be used to 
 *     endorse or promote products derived from this software without specific prior written 
 *     permission.
 *   
 *   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS AS IS AND ANY EXPRESS 
 *   OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY 
 *   AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR 
 *   CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL 
 *   DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, 
 *   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER 
 *   IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT 
 *   OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *   
--**/


#include "ElectronicResourceInfoParser.hpp"
#include "ParserErrors.hpp"
#include "ParserWarnings.hpp"
#include "StringUtils.hpp"
#include "RLLineCodes.hpp"

#include <algorithm>
#include <cstdlib>
#include <cassert>

namespace OpenEMBL
{
namespace Phoenix
{

    ElectronicResourceInfoParser::ElectronicResourceInfoParser(
        IParserCtx*                             pParserCtx,
        IItemHandler<ElectronicResourceInfo>*   pItemHandler)
        : ItemParserImpl<ElectronicResourceInfo>(pParserCtx, pItemHandler)
    {
    }

    METHODIMP ElectronicResourceInfoParser::parse()
    {
        long   theLineNumber = currentLineNumber();
        LPCSTR theLine       = currentLine();

        if (NULL == theLine)
        {
            logError(theLineNumber,
                     ERROR_EOF_FOUND,
                     NULL);
            return;
        }

        if (0 != strNCaseCmp(theLine, RL_ELECTRONIC_RESOURCE, strlen(RL_ELECTRONIC_RESOURCE)))
        {
            logError(theLineNumber,
                     ERROR_INVALID_RL_ELECTRONIC_RESOURCE_LINE,
                     theLine);
            nextLine();
            return;
        }

        StringList theLines;
        extractLines(isRLLine, theLines);
        processLines(theLines, theLineNumber);
    }

    bool ElectronicResourceInfoParser::isRLLine(const char* theLine)
    {
        assert(NULL != theLine);
        return 0 == strncmp(theLine, "RL   ", 5);
    }

    String ElectronicResourceInfoParser::getLineContent(const String & theLine)
    {
        assert(theLine.length() >= 5);
        return trimRight(theLine.substr(5));
    }

    void ElectronicResourceInfoParser::processLines(const StringList & theLines, long theLineNumber)
    {
        ElectronicResourceInfo theField;

        if (!processHeader(theLines[0], theLineNumber, theField))
            return;

        if (!processContent(theLines, theLineNumber + 1, theField))
            return;

        notifyParsed(theLineNumber, theField);
    }

    bool ElectronicResourceInfoParser::processHeader(
        String const &      theLine, 
        long                theLineNumber, 
        ElectronicResourceInfo &    theField)
    {
        std::string theContent = trim(theLine.substr(strlen(RL_ELECTRONIC_RESOURCE)));

        theField.FreeText.push_back(theContent);

        return true;
    }

    bool ElectronicResourceInfoParser::processContent(
        StringList const &  theLines, 
        long                theLineNumber, 
        ElectronicResourceInfo &    theField)
    {
        for (size_t i = 1; i < theLines.size(); ++i)
            theField.FreeText.push_back(getLineContent(theLines[i]));

        return true;
    }

}
}
