/**++
 *   
 *   LICENSE
 *   -------
 *   
 *   Copyright (c) 2004 Renato Mancuso
 *   All rights reserved.
 *   
 *   Redistribution and use in source and binary forms, with or without modification, are 
 *   permitted provided that the following conditions are met:
 *   
 *   - Redistributions of source code must retain the above copyright notice, this list 
 *     of conditions and the following disclaimer.
 *   
 *   - Redistributions in binary form must reproduce the above copyright notice, this list
 *     of conditions and the following disclaimer in the documentation and/or other materials 
 *     provided with the distribution.
 *   
 *   - Neither the name of Renato Mancuso nor the names of its contributors may be used to 
 *     endorse or promote products derived from this software without specific prior written 
 *     permission.
 *   
 *   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS AS IS AND ANY EXPRESS 
 *   OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY 
 *   AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR 
 *   CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL 
 *   DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, 
 *   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER 
 *   IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT 
 *   OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *   
--**/


#include "EntryParser.hpp"
#include "ParserWarnings.hpp"
#include "ParserErrors.hpp"
#include "FieldCodes.hpp"
#include "CCLineParser.hpp"
#include "SQLineParser.hpp"
#include "DRLineParser.hpp"
#include "DELineParser.hpp"
#include "SVLineParser.hpp"
#include "IDLineParser.hpp"
#include "DTLineParser.hpp"
#include "ACLineParser.hpp"
#include "KWLineParser.hpp"
#include "ACStarLineParser.hpp"
#include "TaxonomyLineParser.hpp"
#include "PublicationLineParser.hpp"
#include "SequenceParser.hpp"
#include "FeatureTableParser.hpp"
#include "AssemblyInfoParser.hpp"
#include "ContigInfoParser.hpp"

#include <cstring>
#include <cassert>


namespace OpenEMBL
{
namespace Phoenix
{

    /* class static*/
    EntryParser::HandlerRec EntryParser::HandlerInfo[] = 
    {
        // Line code    Field Code                  Handler
        // (2+3)        (Group)
        { "XX"   ,      EMBL_SPACER_LINE,           &EntryParser::XXLineHandler },
        { "CC   ",      EMBL_COMMENT_LINE,          &EntryParser::CCLineHandler },
        { "DR   ",      EMBL_XREF_LINE,             &EntryParser::DRLineHandler },
        { "RN   ",      EMBL_PUBLICATION_LINE,      &EntryParser::RNLineHandler },
        { "OS   ",      EMBL_TAXONOMY_LINE,         &EntryParser::OSLineHandler },
        { "ID   ",      EMBL_ID_LINE,               &EntryParser::IDLineHandler },
        { "AC   ",      EMBL_AC_LINE,               &EntryParser::ACLineHandler },
        { "SV   ",      EMBL_SV_LINE,               &EntryParser::SVLineHandler },
        { "DE   ",      EMBL_DE_LINE,               &EntryParser::DELineHandler },
        { "DT   ",      EMBL_DT_LINE,               &EntryParser::DTLineHandler },
        { "KW   ",      EMBL_KW_LINE,               &EntryParser::KWLineHandler },
        { "SQ   ",      EMBL_SQ_LINE,               &EntryParser::SQLineHandler },        
        { "     ",      EMBL_SEQUENCE_LINE,         &EntryParser::SequenceHandler },
        { "FT   ",      EMBL_FEATURE_LINE,          &EntryParser::FTLineHandler },
        { "AS   ",      EMBL_ASSEMBLY_INFO_LINE,    &EntryParser::ASLineHandler },
        { "FH"   ,      EMBL_FEATURE_HEADER_LINE,   &EntryParser::nullHandler },
        { "AH"   ,      EMBL_ASSEMBLY_HEADER_LINE,  &EntryParser::nullHandler },
        { "CO   ",      EMBL_CO_LINE,               &EntryParser::COLineHandler },
        { "AC * ",      EMBL_GP_INFO_LINE,          &EntryParser::ACStarLineHandler },
        { "OC   ",      EMBL_TAXONOMY_LINE,         &EntryParser::OCLineHandler },
        { "OG   ",      EMBL_TAXONOMY_LINE,         &EntryParser::OGLineHandler },
        { "RC   ",      EMBL_PUBLICATION_LINE,      &EntryParser::RCLineHandler },
        { "RP   ",      EMBL_PUBLICATION_LINE,      &EntryParser::RPLineHandler },
        { "RX   ",      EMBL_PUBLICATION_LINE,      &EntryParser::RXLineHandler },
        { "RA   ",      EMBL_PUBLICATION_LINE,      &EntryParser::RALineHandler },
        { "RG   ",      EMBL_PUBLICATION_LINE,      &EntryParser::RGLineHandler },
        { "RT   ",      EMBL_PUBLICATION_LINE,      &EntryParser::RTLineHandler },
        { "RL   ",      EMBL_PUBLICATION_LINE,      &EntryParser::RLLineHandler },
        { 0,            0,                          0 }
    };

    /* class static*/
    EntryParser::UsageRec EntryParser::UsageInfo[] =
    {            
        { EMBL_ID_LINE, 1 },
        { EMBL_AC_LINE, 1 },
        { EMBL_SV_LINE, 1 },
        { EMBL_DE_LINE, 1 },
        { EMBL_DT_LINE, 1 },
        { EMBL_KW_LINE, 1 },
        { EMBL_SQ_LINE, 1 },
        { EMBL_SEQUENCE_LINE, 1 },            
        { EMBL_FEATURE_LINE, 1 },
        { EMBL_CO_LINE, 1 },
        { EMBL_GP_INFO_LINE, 1 },
        { EMBL_ASSEMBLY_INFO_LINE, 1 },
        { NULL, 0 }
    };

    EntryParser::EntryParser(
        ITextSource*            theTextSource, 
        IParserEventsSource*    theParserEventsSource)
        : m_TextSource(theTextSource)
        , m_ParserEventsSource(theParserEventsSource)
    {
        initialize();
    }

    EntryParser::~EntryParser()
    {
    }

    void EntryParser::initialize()
    {
        m_Dispatcher.setTargetInstance(this);
        m_Dispatcher.registerStartHandler(&EntryParser::onBeginEntry);
        m_Dispatcher.registerTerminationHandler(&EntryParser::onEndEntry);        

        for (int i = 0; ; ++i)
        {
            HandlerRec & hr = HandlerInfo[i];

            if (NULL == hr.Handler)
                break;

            m_Dispatcher.registerHandler(hr.Header, hr.FieldCode, hr.Handler);
        }

        for (int i = 0; ; ++i)
        {
            UsageRec & ur = UsageInfo[i];

            if (0 == ur.Limit)
                break;

            m_Dispatcher.setUsageLimit(ur.FieldCode, ur.Limit);
        }
    }

    METHODIMP_(LPCSTR) EntryParser::currentLine() const NO_THROW
    {
        if (NULL == m_TextSource)
            return NULL;

        return m_TextSource->currentLine();
    }

    METHODIMP_(long) EntryParser::currentLineNumber() const NO_THROW
    {
        if (NULL == m_TextSource)
            return TEXTSOURCE_INVALID_LINE_NUMBER;

        return m_TextSource->currentLineNumber();
    }

    METHODIMP_(bool) EntryParser::nextLine()
    {
        if (NULL == m_TextSource)
            return false;

        return m_TextSource->nextLine();
    }

    METHODIMP EntryParser::logError(long theLineNumber, int theErrorCode, char const * theMessage)
    {
        broadcastError(theLineNumber, theErrorCode, theMessage);
    }

    METHODIMP EntryParser::logWarning(long theLineNumber, int theWarning, char const * theMessage)
    {
        broadcastWarning(theLineNumber, theWarning, theMessage);
    }

    bool EntryParser::isStartOfRecord(char const* theLine) const
    {
        assert(NULL != theLine);
        return 0 == strncmp(theLine, "ID   ", 5);
    }

    bool EntryParser::isEndOfRecord(char const* theLine) const
    {
        assert(NULL != theLine);
        return 0 == strncmp(theLine, "//", 2);
    }

    void EntryParser::setIgnoreField(std::string const & theFieldCode)
    {
        m_IgnoreFields.insert(theFieldCode);
    }

    bool EntryParser::ignoreField(char const* theFieldCode) const
    {
        assert(NULL != theFieldCode);
        return m_IgnoreFields.find(theFieldCode) != m_IgnoreFields.end();
    }

    char const* EntryParser::getStartOfRecordHeader() const
    {
        return "ID   ";
    }

    METHODIMP EntryParser::parse()
    {
        if (!m_Dispatcher.dispatch())
        {
            if (NULL != currentLine())
            {
                logWarning(currentLineNumber(),
                           WARNING_RESET_PARSER_INPUT,
                           NULL);
                m_Dispatcher.goToNextRecord();
            }
        }
    }

    void EntryParser::onBeginEntry()
    {
        broadcastBeginEntry(currentLineNumber());
    }

    void EntryParser::onEndEntry()
    {
        if (NULL != currentLine())
            TerminatorLineHandler();

        broadcastEndEntry();

        // remove empty lines at end of record
        while (1)
        {
            if (NULL == currentLine())
                break;

            if (0 != strcmp(currentLine(), ""))
                break;

            logWarning(currentLineNumber(),
                       WARNING_EMPTY_LINE_AT_END_OF_RECORD,
                       NULL);

            nextLine();
        }
    }

    void EntryParser::nullHandler()
    {
        nextLine();
    }

    void EntryParser::TerminatorLineHandler()
    {    
        assert(0 == strncmp(currentLine(), "//", 2));

        long        theLineNumber = currentLineNumber();
        std::string theLine       = currentLine();        

        if (theLine.find_first_not_of(" \t", 2) != std::string::npos)
        {
            logWarning(theLineNumber,
                       WARNING_SPURIOUS_CHARACTER_IN_TERMINATOR_LINE,
                       theLine.c_str());
        } 

        nextLine();
    }

    void EntryParser::XXLineHandler()
    {
        assert(NULL != currentLine());
        assert(0 == strncmp(currentLine(), "XX", 2));

        long        theLineNumber = currentLineNumber();
        std::string theLine       = currentLine();        

        if (theLine.find_first_not_of(" \t", 2) != std::string::npos)
        {
            logWarning(theLineNumber,
                       WARNING_SPURIOUS_CHARACTER_IN_XX_LINE,
                       theLine.c_str());
        } 

        broadcastXX(theLineNumber, XXLine());

        nextLine();
    }

    void EntryParser::CCLineHandler()
    {
        CCLineParser theParser(this, this);
        theParser.parse();
    }

    void EntryParser::SequenceHandler()
    {
        SequenceParser theParser(this, this);
        theParser.parse();
    }

    void EntryParser::SQLineHandler()
    {
        SQLineParser theParser(this, this);
        theParser.parse();
    }

    void EntryParser::DRLineHandler()
    {
        DRLineParser theParser(this, this);
        theParser.parse();
    }

    void EntryParser::DELineHandler()
    {
        DELineParser theParser(this, this);
        theParser.parse();
    }

    void EntryParser::SVLineHandler()
    {
        SVLineParser theParser(this, this);
        theParser.parse();
    }

    void EntryParser::IDLineHandler()
    {
        IDLineParser theParser(this, this);
        theParser.parse();
    }

    void EntryParser::DTLineHandler()
    {
        DTLineParser theParser(this, this);
        theParser.parse();
    }

    void EntryParser::ACLineHandler()
    {
        ACLineParser theParser(this, this);
        theParser.parse();
    }

    void EntryParser::KWLineHandler()
    {
        KWLineParser theParser(this, this);
        theParser.parse();
    }

    void EntryParser::OSLineHandler()
    {
        TaxonomyLineParser theParser(this, this);
        theParser.parse();
    }

    void EntryParser::ACStarLineHandler()
    {
        ACStarLineParser theParser(this, this);
        theParser.parse();
    }

    void EntryParser::COLineHandler()
    {
        ContigInfoParser theParser(this, this);
        theParser.parse();
    }

    void EntryParser::OCLineHandler()
    {
        // OC lines are parsed by the TaxonomyLineParser
        logError(currentLineNumber(),
                 ERROR_MISSING_OS_LINE,
                 currentLine());

        nextLine();
    }

    void EntryParser::OGLineHandler()
    {
        // OG lines are parsed by the TaxonomyLineParser
        logError(currentLineNumber(),
                 ERROR_MISSING_OS_LINE,
                 currentLine());

        nextLine();
    }

    void EntryParser::RNLineHandler()
    {
        PublicationLineParser theParser(this, this);
        theParser.parse();
    }

    void EntryParser::RCLineHandler()
    {
        // Publication lines are parsed by the RNLineParser
        logError(currentLineNumber(),
                 ERROR_MISSING_RN_LINE,
                 currentLine());

        nextLine();
    }

    void EntryParser::RPLineHandler()
    {
        // Publication lines are parsed by the RNLineParser
        logError(currentLineNumber(),
                 ERROR_MISSING_RN_LINE,
                 currentLine());

        nextLine();
    }

    void EntryParser::RXLineHandler()
    {
        // Publication lines are parsed by the RNLineParser
        logError(currentLineNumber(),
                 ERROR_MISSING_RN_LINE,
                 currentLine());

        nextLine();
    }

    void EntryParser::RALineHandler()
    {
        // Publication lines are parsed by the RNLineParser
        logError(currentLineNumber(),
                 ERROR_MISSING_RN_LINE,
                 currentLine());

        nextLine();
    }

    void EntryParser::RGLineHandler()
    {
        // Publication lines are parsed by the RNLineParser
        logError(currentLineNumber(),
                 ERROR_MISSING_RN_LINE,
                 currentLine());

        nextLine();
    }

    void EntryParser::RTLineHandler()
    {
        // Publication lines are parsed by the RNLineParser
        logError(currentLineNumber(),
                 ERROR_MISSING_RN_LINE,
                 currentLine());

        nextLine();
    }

    void EntryParser::RLLineHandler()
    {
        // Publication lines are parsed by the RNLineParser
        logError(currentLineNumber(),
                 ERROR_MISSING_RN_LINE,
                 currentLine());

        nextLine();
    }

    void EntryParser::FTLineHandler()
    {
        FeatureTableParser theParser(this, this);        
        theParser.parse();
    }

    void EntryParser::ASLineHandler()
    {
        AssemblyInfoParser theParser(this, this);
        theParser.parse();
    }

    METHODIMP EntryParser::onParsed(long theLineNumber, const CCLine & theItem)
    {
        broadcastCC(theLineNumber, theItem);
    }

    METHODIMP EntryParser::onParsed(long theLineNumber, const SequenceLine & theItem)
    {
        broadcastSequenceLine(theLineNumber, theItem);
    }

    METHODIMP EntryParser::onParsed(long theLineNumber, const SQLine & theItem)
    {
        broadcastSQ(theLineNumber, theItem);
    }

    METHODIMP EntryParser::onParsed(long theLineNumber, const DRLine & theItem)
    {
        broadcastDR(theLineNumber, theItem);
    }

    METHODIMP EntryParser::onParsed(long theLineNumber, const DELine & theItem)
    {
        broadcastDE(theLineNumber, theItem);
    }

    METHODIMP EntryParser::onParsed(long theLineNumber, const SVLine & theItem)
    {
        broadcastSV(theLineNumber, theItem);
    }

    METHODIMP EntryParser::onParsed(long theLineNumber, const IDLine & theItem)
    {
        broadcastID(theLineNumber, theItem);
    }

    METHODIMP EntryParser::onParsed(long theLineNumber, const DTLine & theItem)
    {
        broadcastDT(theLineNumber, theItem);
    }

    METHODIMP EntryParser::onParsed(long theLineNumber, const ACLine & theItem)
    {
        broadcastAC(theLineNumber, theItem);
    }

    METHODIMP EntryParser::onParsed(long theLineNumber, const KWLine & theItem)
    {
        broadcastKW(theLineNumber, theItem);
    }

    METHODIMP EntryParser::onParsed(long theLineNumber, const OSLine & theItem)
    {
        broadcastOS(theLineNumber, theItem);
    }

    METHODIMP EntryParser::onParsed(long theLineNumber, const OCLine & theItem)
    {
        broadcastOC(theLineNumber, theItem);
    }

    METHODIMP EntryParser::onParsed(long theLineNumber, const OGLine & theItem)
    {
        broadcastOG(theLineNumber, theItem);
    }

    METHODIMP EntryParser::onParsed(long theLineNumber, const RNLine & theItem)
    {
        broadcastRN(theLineNumber, theItem);
    }

    METHODIMP EntryParser::onParsed(long theLineNumber, const RCLine & theItem)
    {
        broadcastRC(theLineNumber, theItem);
    }

    METHODIMP EntryParser::onParsed(long theLineNumber, const RPLine & theItem)
    {
        broadcastRP(theLineNumber, theItem);
    }

    METHODIMP EntryParser::onParsed(long theLineNumber, const RXLine & theItem)
    {
        broadcastRX(theLineNumber, theItem);
    }

    METHODIMP EntryParser::onParsed(long theLineNumber, const RALine & theItem)
    {
        broadcastRA(theLineNumber, theItem);
    }

    METHODIMP EntryParser::onParsed(long theLineNumber, const RGLine & theItem)
    {
        broadcastRG(theLineNumber, theItem);
    }

    METHODIMP EntryParser::onParsed(long theLineNumber, const RTLine & theItem)
    {
        broadcastRT(theLineNumber, theItem);
    }

    METHODIMP EntryParser::onParsed(long theLineNumber, const SubmissionInfo & theItem)
    {
        broadcastSubmission(theLineNumber, theItem);
    }

    METHODIMP EntryParser::onParsed(long theLineNumber, const UnpublishedInfo & theItem)
    {
        broadcastUnpublished(theLineNumber, theItem);
    }

    METHODIMP EntryParser::onParsed(long theLineNumber, const BookInfo & theItem)
    {
        broadcastBook(theLineNumber, theItem);
    }

    METHODIMP EntryParser::onParsed(long theLineNumber, const ThesisInfo & theItem)
    {
        broadcastThesis(theLineNumber, theItem);
    }

    METHODIMP EntryParser::onParsed(long theLineNumber, const PatentInfo & theItem)
    {
        broadcastPatent(theLineNumber, theItem);
    }

    METHODIMP EntryParser::onParsed(long theLineNumber, const OnlineJournalInfo & theItem)
    {
        broadcastOnlineJournal(theLineNumber, theItem);
    }

    METHODIMP EntryParser::onParsed(long theLineNumber, const ElectronicResourceInfo & theItem)
    {
        broadcastElectronicResource(theLineNumber, theItem);
    }

    METHODIMP EntryParser::onParsed(long theLineNumber, const MiscResourceInfo & theItem)
    {
        broadcastMiscResource(theLineNumber, theItem);
    }

	METHODIMP EntryParser::onParsed(long theLineNumber, const JournalArticleInfo & theItem)
    {
        broadcastJournalArticle(theLineNumber, theItem);
    }

    METHODIMP EntryParser::onBeginPublication(long theLineNumber)
    {
        broadcastBeginPublication(theLineNumber);
    }

    METHODIMP EntryParser::onEndPublication()
    {
        broadcastEndPublication();
    }

    METHODIMP EntryParser::onBeginTaxonomy(long theLineNumber)
    {
        broadcastBeginTaxonomyLines(theLineNumber);
    }

    METHODIMP EntryParser::onEndTaxonomy()
    {
        broadcastEndTaxonomyLines();
    }

    METHODIMP EntryParser::onBeginSequence(long theLineNumber)
    {
        broadcastBeginSequenceData(theLineNumber);
    }

    METHODIMP EntryParser::onEndSequence()
    {
        broadcastEndSequenceData();
    }

    METHODIMP EntryParser::onBeginFeatureTable(long theLineNumber)
    {
        broadcastBeginFeatureTable(theLineNumber);
    }

    METHODIMP EntryParser::onEndFeatureTable()
    {
        broadcastEndFeatureTable();
    }

    METHODIMP EntryParser::onParsed(long theLineNumber, const FeatureKey & theItem)
    {
        broadcastBeginFeatureKey(theLineNumber, theItem);
    }

    METHODIMP EntryParser::onEndFeatureKey()
    {
        broadcastEndFeatureKey();
    }

    METHODIMP EntryParser::onParsed(long theLineNumber, const FeatureQualifier & theItem)
    {
        broadcastFeatureQualifier(theLineNumber, theItem);
    }

    METHODIMP EntryParser::onBeginAssemblyInfo(long theLineNumber)
    {
        broadcastBeginTPALines(theLineNumber);
    }

    METHODIMP EntryParser::onEndAssemblyInfo()
    {
        broadcastEndTPALines();
    }

    METHODIMP EntryParser::onParsed(long theLineNumber, const ASLine & theItem)
    {
        broadcastAS(theLineNumber, theItem);
    }

    METHODIMP EntryParser::onParsed(long theLineNumber, const ACStarLine & theItem)
    {
        broadcastACStar(theLineNumber, theItem);
    }

    METHODIMP EntryParser::onBeginContigInfo(long theLineNumber, bool isComplement)
    {
        broadcastBeginCOLines(theLineNumber, isComplement);
    }

    METHODIMP EntryParser::onEndContigInfo()
    {
        broadcastEndCOLines();
    }

    METHODIMP EntryParser::onParsed(long theLineNumber, const COSegmentInfo & theItem)
    {
        broadcastCOSegmentInfo(theLineNumber, theItem);
    }

    METHODIMP EntryParser::onParsed(long theLineNumber, const COGapInfo & theItem)
    {
        broadcastCOGapInfo(theLineNumber, theItem);
    }

}
} 
