/**++
 *   
 *   LICENSE
 *   -------
 *   
 *   Copyright (c) 2004 Renato Mancuso
 *   All rights reserved.
 *   
 *   Redistribution and use in source and binary forms, with or without modification, are 
 *   permitted provided that the following conditions are met:
 *   
 *   - Redistributions of source code must retain the above copyright notice, this list 
 *     of conditions and the following disclaimer.
 *   
 *   - Redistributions in binary form must reproduce the above copyright notice, this list
 *     of conditions and the following disclaimer in the documentation and/or other materials 
 *     provided with the distribution.
 *   
 *   - Neither the name of Renato Mancuso nor the names of its contributors may be used to 
 *     endorse or promote products derived from this software without specific prior written 
 *     permission.
 *   
 *   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS AS IS AND ANY EXPRESS 
 *   OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY 
 *   AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR 
 *   CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL 
 *   DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, 
 *   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER 
 *   IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT 
 *   OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *   
--**/


#ifndef OPEN_EMBL_ENTRY_PARSER_INCLUDED
#define OPEN_EMBL_ENTRY_PARSER_INCLUDED

#if !defined(OPEN_EMBL_NO_PRAGMA_ONCE)
#pragma once
#endif

#include <set>
#include <string>

#include "NonCopyable.hpp"
#include "ParserItf.hpp"
#include "ParserEventsSource.hpp"
#include "HandlerDispatch.hpp"
#include "ParserCtx.hpp"
#include "FieldDefs.hpp"
#include "ItemHandler.hpp"
#include "TaxonomyHandler.hpp"
#include "PublicationHandler.hpp"
#include "SequenceHandler.hpp"
#include "FeatureTableHandler.hpp"
#include "AssemblyInfoHandler.hpp"
#include "ContigInfoHandler.hpp"

namespace OpenEMBL
{
namespace Phoenix
{

    class EntryParser
        : public  virtual IParser
        , private virtual IParserCtx
        , private virtual IParserEventsSource
        , private virtual IItemHandler<IDLine>
        , private virtual IItemHandler<SVLine>
        , private virtual IItemHandler<DTLine>
        , private virtual IItemHandler<DELine>
        , private virtual IItemHandler<CCLine>
        , private virtual IItemHandler<DRLine>
        , private virtual IItemHandler<SQLine>
        , private virtual IItemHandler<ACLine>
        , private virtual IItemHandler<KWLine>
        , private virtual IItemHandler<ACStarLine>
        , private virtual ITaxonomyHandler
        , private virtual IPublicationHandler
        , private virtual ISequenceHandler
        , private virtual IFeatureTableHandler
        , private virtual IAssemblyInfoHandler
        , private virtual IContigInfoHandler
        , NonCopyable
    {
    public:
        EntryParser(ITextSource* theTextSource, IParserEventsSource* theParserEventsSource);
        ~EntryParser();

        // Parser
        METHOD parse();        

    private:
        // methods
        void         setIgnoreField(std::string const & theFieldCode);
        
        // methods required by DispatchHandler
        void         initialize();
        bool         isStartOfRecord (char const* theLine) const;        
        bool         isEndOfRecord   (char const* theLine) const;
        bool         ignoreField     (char const* theFieldCode) const;
        char const*  getStartOfRecordHeader() const;        

        // TextSource (ParserCtx)
        METHOD_(LPCSTR) currentLine() const NO_THROW;
        METHOD_(long)   currentLineNumber() const NO_THROW;
        METHOD_(bool)   nextLine();

        // ErrorLog (ParserCtx)
        METHOD logError(long theLineNumber, int theErrorCode, LPCSTR theMessage);
        METHOD logWarning(long theLineNumber, int theWarning, LPCSTR theMessage);

        // ParserEventsSource
        METHOD broadcastError  (long theLineNumber, int theErrorCode, char const* theMessage);
        METHOD broadcastWarning(long theLineNumber, int theWarningNumber, char const* theMessage);
        METHOD broadcastBeginEntry(long theLineNumber);
        METHOD broadcastEndEntry();
        METHOD broadcastID(long theLineNumber, IDLine const & theField);
        METHOD broadcastAC(long theLineNumber, ACLine const & theField);
        METHOD broadcastSV(long theLineNumber, SVLine const & theField);
        METHOD broadcastDT(long theLineNumber, DTLine const & theField);
        METHOD broadcastDE(long theLineNumber, DELine const & theField);
        METHOD broadcastKW(long theLineNumber, KWLine const & theField);
        METHOD broadcastDR(long theLineNumber, DRLine const & theField);
        METHOD broadcastCC(long theLineNumber, CCLine const & theField);
        METHOD broadcastSQ(long theLineNumber, SQLine const & theField);
        METHOD broadcastXX(long theLineNumber, XXLine const & theField);
        METHOD broadcastBeginTaxonomyLines(long theLineNumber);
        METHOD broadcastOS(long theLineNumber, OSLine const & theField);    
        METHOD broadcastOC(long theLineNumber, OCLine const & theField);
        METHOD broadcastOG(long theLineNumber, OGLine const & theField);        
        METHOD broadcastEndTaxonomyLines();
        METHOD broadcastBeginFeatureTable(long theLineNumber);
        METHOD broadcastBeginFeatureKey(long theLineNumber, FeatureKey const & theKey);
        METHOD broadcastFeatureQualifier(long theLineNumber, FeatureQualifier const & theQualifier);
        METHOD broadcastEndFeatureKey();        
        METHOD broadcastEndFeatureTable();
        METHOD broadcastBeginSequenceData(long theLineNumber);
        METHOD broadcastSequenceLine(long theLineNumber, SequenceLine const & theField);
        METHOD broadcastEndSequenceData();
        METHOD broadcastBeginTPALines(long theLineNumber);
        METHOD broadcastAS(long theLineNumber, ASLine const & theField);
        METHOD broadcastEndTPALines();
        METHOD broadcastBeginCOLines(long theLineNumber, bool isComplement);        
        METHOD broadcastCOSegmentInfo (long theLineNumber, COSegmentInfo const & theSegmentInfo);
        METHOD broadcastCOGapInfo (long theLineNumber, COGapInfo const & theGapInfo);
        METHOD broadcastEndCOLines();
        METHOD broadcastACStar(long theLineNumber, ACStarLine const & theField);
        METHOD broadcastBeginPublication(long theLineNumber);
        METHOD broadcastRN(long theLineNumber, RNLine const & theField);
        METHOD broadcastRC(long theLineNumber, RCLine const & theField);
        METHOD broadcastRP(long theLineNumber, RPLine const & theField);
        METHOD broadcastRX(long theLineNumber, RXLine const & theField);
        METHOD broadcastRG(long theLineNumber, RGLine const & theField);
        METHOD broadcastRA(long theLineNumber, RALine const & theField);
        METHOD broadcastRT(long theLineNumber, RTLine const & theField);
        METHOD broadcastSubmission        (long theLineNumber, SubmissionInfo         const & theField);
        METHOD broadcastUnpublished       (long theLineNumber, UnpublishedInfo        const & theField);
        METHOD broadcastBook              (long theLineNumber, BookInfo               const & theField);
        METHOD broadcastThesis            (long theLineNumber, ThesisInfo             const & theField);
        METHOD broadcastPatent            (long theLineNumber, PatentInfo             const & theField);
        METHOD broadcastOnlineJournal     (long theLineNumber, OnlineJournalInfo      const & theField);
        METHOD broadcastElectronicResource(long theLineNumber, ElectronicResourceInfo const & theField);
		METHOD broadcastMiscResource      (long theLineNumber, MiscResourceInfo       const & theField);
        METHOD broadcastJournalArticle    (long theLineNumber, JournalArticleInfo     const & theField);
        METHOD broadcastEndPublication();

        // handlers
        void onBeginEntry();
        void onEndEntry();
        void nullHandler();
        void TerminatorLineHandler();
        void XXLineHandler();
        void CCLineHandler();
        void SequenceHandler();
        void SQLineHandler();
        void DRLineHandler();
        void DELineHandler();
        void SVLineHandler();
        void IDLineHandler();
        void DTLineHandler();  
        void ACLineHandler();
        void KWLineHandler();
        void OSLineHandler();
        void OCLineHandler();
        void OGLineHandler();
        void RNLineHandler();
        void RCLineHandler();
        void RPLineHandler();
        void RXLineHandler();
        void RALineHandler();
        void RGLineHandler();
        void RTLineHandler();
        void RLLineHandler();
        void FTLineHandler();
        void ASLineHandler();
        void ACStarLineHandler();
        void COLineHandler();

        // item notifications
        METHOD onParsed(long theLineNumber, const IDLine & theItem);
        METHOD onParsed(long theLineNumber, const CCLine & theItem);
        METHOD onParsed(long theLineNumber, const DELine & theItem);
        METHOD onParsed(long theLineNumber, const DRLine & theItem);        
        METHOD onParsed(long theLineNumber, const SQLine & theItem);
        METHOD onParsed(long theLineNumber, const SVLine & theItem);
        METHOD onParsed(long theLineNumber, const DTLine & theItem);
        METHOD onParsed(long theLineNumber, const ACLine & theItem);
        METHOD onParsed(long theLineNumber, const KWLine & theItem);
        METHOD onParsed(long theLineNumber, const ACStarLine & theItem);

        METHOD onBeginTaxonomy(long theLineNumber);
        METHOD onParsed(long theLineNumber, const OSLine & theItem);
        METHOD onParsed(long theLineNumber, const OCLine & theItem);
        METHOD onParsed(long theLineNumber, const OGLine & theItem); 
        METHOD onEndTaxonomy();

        METHOD onBeginPublication(long theLineNumber);
        METHOD onParsed(long theLineNumber, const RNLine & theItem);
        METHOD onParsed(long theLineNumber, const RCLine & theItem);
        METHOD onParsed(long theLineNumber, const RPLine & theItem);
        METHOD onParsed(long theLineNumber, const RXLine & theItem);
        METHOD onParsed(long theLineNumber, const RALine & theItem);
        METHOD onParsed(long theLineNumber, const RGLine & theItem);
        METHOD onParsed(long theLineNumber, const RTLine & theItem);
        METHOD onParsed(long theLineNumber, const SubmissionInfo & theItem);
        METHOD onParsed(long theLineNumber, const UnpublishedInfo & theItem);
        METHOD onParsed(long theLineNumber, const BookInfo & theItem);
        METHOD onParsed(long theLineNumber, const ThesisInfo & theItem);
        METHOD onParsed(long theLineNumber, const PatentInfo & theItem);
        METHOD onParsed(long theLineNumber, const OnlineJournalInfo & theItem);
        METHOD onParsed(long theLineNumber, const ElectronicResourceInfo & theItem);
		METHOD onParsed(long theLineNumber, const MiscResourceInfo & theItem);
        METHOD onParsed(long theLineNumber, const JournalArticleInfo & theItem);
        METHOD onEndPublication();

        METHOD onBeginSequence(long theLineNumber);
        METHOD onParsed(long theLineNumber, const SequenceLine & theItem);
        METHOD onEndSequence();

        METHOD onBeginFeatureTable(long theLineNumber);
        METHOD onParsed(long theLineNumber, const FeatureKey & theItem);
        METHOD onParsed(long theLineNumber, const FeatureQualifier & theItem);
        METHOD onEndFeatureKey();
        METHOD onEndFeatureTable();

        METHOD onBeginAssemblyInfo(long theLineNumber);
        METHOD onParsed(long theLineNumber, const ASLine & theItem);
        METHOD onEndAssemblyInfo();        

        METHOD onBeginContigInfo(long theLineNumber, bool isComplement);
        METHOD onParsed(long theLineNumber, const COSegmentInfo & theItem);
        METHOD onParsed(long theLineNumber, const COGapInfo & theItem);
        METHOD onEndContigInfo();

    private:
        struct HandlerRec
        {
            char const*     Header;
            char const*     FieldCode;
            void (EntryParser::*Handler)(void);
        };

        struct UsageRec 
        {
            char const*     FieldCode;
            size_t          Limit;
        };

        static HandlerRec HandlerInfo[];
        static UsageRec   UsageInfo[];

    private:
        friend class HandlerDispatch<EntryParser>;

        ITextSource*                    m_TextSource;
        IParserEventsSource*            m_ParserEventsSource;
        HandlerDispatch<EntryParser>    m_Dispatcher;
        std::set<std::string>           m_IgnoreFields;
    };
}
} 

#endif // OPEN_EMBL_ENTRY_PARSER_INCLUDED
