/**++
 *   
 *   LICENSE
 *   -------
 *   
 *   Copyright (c) 2004 Renato Mancuso
 *   All rights reserved.
 *   
 *   Redistribution and use in source and binary forms, with or without modification, are 
 *   permitted provided that the following conditions are met:
 *   
 *   - Redistributions of source code must retain the above copyright notice, this list 
 *     of conditions and the following disclaimer.
 *   
 *   - Redistributions in binary form must reproduce the above copyright notice, this list
 *     of conditions and the following disclaimer in the documentation and/or other materials 
 *     provided with the distribution.
 *   
 *   - Neither the name of Renato Mancuso nor the names of its contributors may be used to 
 *     endorse or promote products derived from this software without specific prior written 
 *     permission.
 *   
 *   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS AS IS AND ANY EXPRESS 
 *   OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY 
 *   AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR 
 *   CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL 
 *   DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, 
 *   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER 
 *   IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT 
 *   OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *   
--**/


#include "FeatureKeyParser.hpp"
#include "ParserErrors.hpp"
#include "ParserWarnings.hpp"
#include "StringUtils.hpp"
#include "TextSourceOnStringListAdapter.hpp"
#include "ParserCtxOnErrorLogAdapter.hpp"
#include "FeatureLocationParser.hpp"
#include "FeatureQualifierParser.hpp"

#include <string>
#include <algorithm>
#include <cstring>
#include <cctype>
#include <cassert>

namespace OpenEMBL
{
namespace Phoenix
{

    FeatureKeyParser::FeatureKeyParser(
        IParserCtx*             theParserCtx,
        IFeatureTableHandler*   theFeatureTableHandler)
        : ItemParserImpl<FeatureKey>(theParserCtx, theFeatureTableHandler)
        , m_FeatureTableHandler(theFeatureTableHandler)
    {
    }

    void FeatureKeyParser::notifyEndFeatureKey()
    {
        if (NULL != m_FeatureTableHandler)
            m_FeatureTableHandler->onEndFeatureKey();
    }

    bool FeatureKeyParser::isFeatureKeyLine(char const * theLine)
    {
        if (NULL == theLine)
            return false;

        std::string theRow = theLine;

        if (theRow.length() < 6)
            return false;

        std::string theKey = trimLeft(theRow.substr(5, 15));
        
        return !theKey.empty();
    }

    void FeatureKeyParser::goToNextKey()
    {
        while (1)
        {
            if (NULL == currentLine())
                break;

            if (isFeatureKeyLine(currentLine()))
                break;

            nextLine();
        }
    }

    void FeatureKeyParser::extractLines(StringList & theLines)
    {
        theLines.push_back(currentLine());
        nextLine();

        while (1)
        {
            if (NULL == currentLine())
                break;

            if (isFeatureKeyLine(currentLine()))
                break;

            theLines.push_back(currentLine());

            nextLine();            
        }
    }

    METHODIMP FeatureKeyParser::parse()
    {
        if (isFeatureKeyLine(currentLine()))
        {
            long       theLineNumber = currentLineNumber();
            StringList theLines;

            extractLines(theLines);
            processLines(theLineNumber, theLines);
        }
        else
        {
            logError(currentLineNumber(),
                     ERROR_MISSING_FEATURE_KEY,
                     currentLine());

            logWarning(currentLineNumber(),
                       WARNING_SKIPPING_LINES_SEARCHING_FOR_FKEY,
                       NULL);

            goToNextKey();
        }
    }

    void FeatureKeyParser::processLines(long theLineNumber, StringList const & theLines)
    {
        assert(theLines.size() > 0);

        if (!validateLayout(theLineNumber, theLines))
            return;

        String     theKeyLine;
        StringList theLocationLines;
        StringList theQualifierLines;

        sortLines(theLines, theKeyLine, theLocationLines, theQualifierLines);

        FeatureKey theKey;

        if (!processFeatureKey(theLineNumber, theKeyLine, theKey.Key))
            return;

        if (!processFeatureLocation(theLineNumber, theLocationLines, theKey.Location))
            return;        

        notifyParsed(theLineNumber, theKey);

        processFeatureQualifiers(theLineNumber + (long) theLocationLines.size(),
                                 theQualifierLines);

        notifyEndFeatureKey();
    }

    bool FeatureKeyParser::validateLayout(long theLineNumber, StringList const & theLines)
    {
        for (size_t i = 0; i < theLines.size(); ++i)
        {
            std::string theLine = theLines[i];

            if (theLine.length() < 22)
            {
                logError(theLineNumber + long(i),
                         ERROR_FT_LINE_TOO_SHORT,
                         theLine.c_str());
                return false;
            }

            std::string theCol1         = theLine.substr(5, 15);
            std::string theColSeparator = theLine.substr(20, 1);
            std::string theCol2         = theLine.substr(21);

            assert(i == 0 ? 
                   trimLeft(theCol1).empty() == false :
                   trimLeft(theCol1).empty() == true);

            if (!isspace(theColSeparator[0]))
            {
                logError(theLineNumber + long(i),
                        ERROR_FT_LINE_MISSING_COLUMN_SEPARATOR,
                        theLine.c_str());
                return false;
            }

            if (trimLeft(theCol2).empty())
            {
                logError(theLineNumber + long(i),
                         ERROR_FT_LINE_EMPTY_DATA_COLUMN,
                         theLine.c_str());
                return false;
            }
        }

        return true;
    }

    void FeatureKeyParser::getKeyColumnContent(
            String const & theLine,
            String &       theValue)
    {
        theValue = theLine.substr(5, 15);
    }

    void FeatureKeyParser::getDataColumnContent(
            String const & theLine, 
            String &       theValue)
    {
        theValue = theLine.substr(21);
    }

    void FeatureKeyParser::sortLines(
        StringList const &  theLines,
        String     &        theKey,
        StringList &        theLocationLines,
        StringList &        theQualifierLines )
    {
        std::string theValue;
        
        getKeyColumnContent(theLines[0], theKey);     
        getDataColumnContent(theLines[0], theValue);

        theLocationLines.push_back(theValue);

        bool inQualifierSection = false;

        for (size_t i = 1; i < theLines.size(); ++i)
        {
            getDataColumnContent(theLines[i], theValue);            

            if (!inQualifierSection)
            {
                if (trimLeft(theValue)[0] == '/')
                    inQualifierSection = true;
            }

            if (inQualifierSection)
                theQualifierLines.push_back(theValue);
            else
                theLocationLines.push_back(theValue);
        }
    }

    bool FeatureKeyParser::processFeatureKey(
                long                theLineNumber, 
                String const &      theKeyLine,
                String &            theKey)
    {
        if (isspace(theKeyLine[0]))
        {
            logWarning(theLineNumber,
                       WARNING_FEATURE_KEY_NOT_IN_PLACE,
                       trim(theKeyLine).c_str());
        }

        theKey = trim(theKeyLine);

        return true;
    }

    bool FeatureKeyParser::processFeatureLocation(
                long                theLineNumber, 
                StringList const &  theLines,
                FeatureLocation &   theLocation)
    {
        TextSourceOnStringListAdapter   theTextSource(theLines, theLineNumber);
        ParserCtxOnErrorLogAdapter      theParserCtx(this, &theTextSource);

        FeatureLocationParser theParser(&theParserCtx);
        return theParser.parse(theLocation);
    }

    void FeatureKeyParser::processFeatureQualifiers(long theLineNumber, StringList const & theLines)
    {
        TextSourceOnStringListAdapter theTextSource(theLines, theLineNumber);
        ParserCtxOnErrorLogAdapter    theParserCtx(this, &theTextSource);

        while (theTextSource.currentLine() != NULL)
        {
            FeatureQualifierParser theParser(&theParserCtx, m_FeatureTableHandler);
            theParser.parse();
        }
    }

}
}
