/**++
 *   
 *   LICENSE
 *   -------
 *   
 *   Copyright (c) 2004 Renato Mancuso
 *   All rights reserved.
 *   
 *   Redistribution and use in source and binary forms, with or without modification, are 
 *   permitted provided that the following conditions are met:
 *   
 *   - Redistributions of source code must retain the above copyright notice, this list 
 *     of conditions and the following disclaimer.
 *   
 *   - Redistributions in binary form must reproduce the above copyright notice, this list
 *     of conditions and the following disclaimer in the documentation and/or other materials 
 *     provided with the distribution.
 *   
 *   - Neither the name of Renato Mancuso nor the names of its contributors may be used to 
 *     endorse or promote products derived from this software without specific prior written 
 *     permission.
 *   
 *   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS AS IS AND ANY EXPRESS 
 *   OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY 
 *   AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR 
 *   CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL 
 *   DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, 
 *   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER 
 *   IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT 
 *   OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *   
--**/


#include "FeatureLocationParser.hpp"
#include "ParserErrors.hpp"
#include "ParserWarnings.hpp"

#include <cstdio>
#include <sstream>
#include <cassert>

namespace OpenEMBL
{
namespace Phoenix
{

    FeatureLocationParser::FeatureLocationParser(IParserCtx* theParserCtx)
        : m_ParserCtx(theParserCtx)
    {
        m_Lexer.setErrorLog(this);
        m_Lexer.setTextSource(this);
    }

    METHODIMP_(LPCSTR) FeatureLocationParser::currentLine() const NO_THROW
    {
        if (NULL != m_ParserCtx)
            return m_ParserCtx->currentLine();

        return NULL;
    }

    METHODIMP_(long)   FeatureLocationParser::currentLineNumber() const NO_THROW
    {
        if (NULL != m_ParserCtx)
            return m_ParserCtx->currentLineNumber();

        return TEXTSOURCE_INVALID_LINE_NUMBER;
    }

    METHODIMP_(bool)   FeatureLocationParser::nextLine()
    {
        if (NULL != m_ParserCtx)
            return m_ParserCtx->nextLine();

        return false;
    }

    METHODIMP FeatureLocationParser::logError(long theLineNumber, int theErrorCode, LPCSTR theMessage)
    {
        if (NULL != m_ParserCtx)
            return m_ParserCtx->logError(theLineNumber, theErrorCode, theMessage);
    }

    METHODIMP FeatureLocationParser::logWarning(long theLineNumber, int theWarning, LPCSTR theMessage)
    {
        if (NULL != m_ParserCtx)
            return m_ParserCtx->logWarning(theLineNumber, theWarning, theMessage);
    }

    std::string FeatureLocationParser::toString(int theToken)
    {
        switch (theToken)
        {
        case EOF:
            return "<END OF LOCATION>";

        case BAD_TOKEN:
            return "<BAD_TOKEN>";

        case ACCESSION_NUMBER:
            return "accession number";

        case NUMBER:
            return "number";

        case DOUBLE_DOT:
            return "..";

        case JOIN:
            return "join";

        case ORDER:
            return "order";

        case COMPLEMENT:
            return "complement";

        case GAP:
            return "gap";

        case COMMA:
            return ",";

        case DOT:
            return ".";

        case COLON:
            return ":";

        case CARET:
            return "^";

        case OPENING_PARENTHESIS:
            return "(";

        case CLOSING_PARENTHESIS:
            return ")";

        case OPENING_ANGLE_BRACKET:
            return "<";

        case CLOSING_ANGLE_BRACKET:
            return ">";

        default:
            assert(0);
        }

        return "<UNKNOWN TOKEN>";
    }

    bool FeatureLocationParser::parse(FeatureLocation & theLocation)
    {
        try
        {
            parseLocation(theLocation);
            return true;
        }
        catch (InvalidLocation &)
        {         
            /* DO NOTHING */;
        }
        return false;
    }

    int FeatureLocationParser::getToken()
    {
        int theToken = m_Lexer.getToken();

        if (EOF == theToken)
        {
            logError(currentLineNumber(),
                     ERROR_UNEXPECTED_LOCATION_END,
                     NULL);

            throw InvalidLocation();
        }

        if (BAD_TOKEN == theToken)
        {
            // message should already have been provided by Lexer
            throw InvalidLocation();
        }

        if (GAP == theToken)
        {
            logError(currentLineNumber(),
                     ERROR_GAP_IN_LOCATION_STRING,
                     NULL);

            throw InvalidLocation();
        }

        return theToken;
    }

    int FeatureLocationParser::peekToken() const throw()
    {
        return m_Lexer.peekToken();
    }

    std::string FeatureLocationParser::getTokenValue() const
    {
        std::string theValue;
        
        if (!m_Lexer.getTokenValue(theValue))
        {
            assert(0);
        }           

        return theValue;
    }

    void FeatureLocationParser::badToken(int theExpectedToken, int theFoundToken)
    {
        std::ostringstream oss;

        oss << "expected '" << toString(theExpectedToken)
            << "' found '"  << getTokenValue()            
            << "' [token type: '" << toString(theFoundToken)
            << "']"
            ;

        logError(currentLineNumber(),
                 ERROR_BAD_TOKEN_IN_LOCATION,
                 oss.str().c_str());

        throw InvalidLocation();
    }

    void FeatureLocationParser::readToken(int whichToken)
    {
        int theToken = getToken();

        if (theToken != whichToken)
            badToken(whichToken, theToken);
    }

    void FeatureLocationParser::readToken(int whichToken, std::string & theValue)
    {
        readToken(whichToken);       
        theValue = getTokenValue();
    }

    void FeatureLocationParser::readAccessionNumber(std::string & theValue)
    {
        readToken(ACCESSION_NUMBER, theValue);
    }

    void FeatureLocationParser::readNumber(std::string & theValue)
    {
        readToken(NUMBER, theValue);
    }

    void FeatureLocationParser::readNumber(long & theValue)
    {
        std::string theRep;
        readNumber(theRep);

        theValue = atol(theRep.c_str());
    }

    void FeatureLocationParser::readDoubleDot()
    {
        readToken(DOUBLE_DOT);
    }

    void FeatureLocationParser::readJoin()
    {
        readToken(JOIN);
    }

    void FeatureLocationParser::readOrder()
    {
        readToken(ORDER);
    }

    void FeatureLocationParser::readComplement()
    {
        readToken(COMPLEMENT);
    }

    void FeatureLocationParser::readComma()
    {
        readToken(COMMA);
    }

    void FeatureLocationParser::readDot()
    {
        readToken(DOT);
    }

    void FeatureLocationParser::readColon()
    {
        readToken(COLON);
    }

    void FeatureLocationParser::readCaret()
    {
        readToken(CARET);
    }

    void FeatureLocationParser::readOpeningParenthesis()
    {
        readToken(OPENING_PARENTHESIS);
    }

    void FeatureLocationParser::readClosingParenthesis()
    {
        readToken(CLOSING_PARENTHESIS);
    }

    void FeatureLocationParser::readOpeningAngleBracket()
    {
        readToken(OPENING_ANGLE_BRACKET);
    }

    void FeatureLocationParser::readClosingAngleBracket()
    {
        readToken(CLOSING_ANGLE_BRACKET);
    }

    void FeatureLocationParser::parseLocation(FeatureLocation & theLocation)
    {
        bool isComplement   = false;
        bool isList         = false;        

        if (peekToken() == COMPLEMENT)
        {
            isComplement = true;
            readComplement();
            readOpeningParenthesis();
            theLocation.IsComplement = true;
        }

        if (peekToken() == JOIN || peekToken() == ORDER)
        {
            isList = true;

            if (peekToken() == JOIN)
            {
                readJoin();
                readOpeningParenthesis();
                theLocation.IsJoin = true;
            }
            else
            {
                readOrder();
                readOpeningParenthesis();
            }
        }

        readElement(theLocation);

        if (isList)
        {
            readComma();
            
            while (1)
            {
                readElement(theLocation);

                if (peekToken() == CLOSING_PARENTHESIS)
                    break;

                readComma();
            }
        }

        if (isList)
            readClosingParenthesis();

        if (isComplement)
            readClosingParenthesis();

        if (peekToken() != EOF)
            badToken(EOF, getToken());
    }

    void FeatureLocationParser::readElement(FeatureLocation & theLocation)
    {
        bool isComplement = false;

        LocationElement theElement;

        if (peekToken() == COMPLEMENT)
        {
            isComplement = true;
            readComplement();
            readOpeningParenthesis();
            theElement.e_flags |= LOCATION_ELEMENT_COMPLEMENT;
        }

        if (peekToken() == ACCESSION_NUMBER)
        {
            readAccessionNumber(theElement.AccessionNumber);

            if (peekToken() == DOT)
            {
                std::string theSequenceVersion;

                readDot();
                readNumber(theSequenceVersion);
                theElement.SequenceVersion = atoi(theSequenceVersion.c_str());
            }

            readColon();
        }

        readBasePosition(theElement.x1, theElement.x2, theElement.x_flags);

        // if .. or ^ then read y
        if (peekToken() == DOUBLE_DOT || peekToken() == CARET)
        {
            if (peekToken() == CARET)
            {
                readCaret();
                theElement.e_flags |= LOCATION_ELEMENT_INFRABASE;                

                //
                // caret is only compatible with simple base position values
                //
                if (theElement.x_flags != BASE_POSITION_SIMPLE_BASE)
                    badToken(DOUBLE_DOT, CARET);                                

                if (peekToken() != NUMBER)
                    badToken(NUMBER, getToken());                
            }
            else
            {
                readDoubleDot();
                theElement.e_flags |= LOCATION_ELEMENT_RANGE;
            }

            readBasePosition(theElement.y1, theElement.y2, theElement.y_flags);
        }
        else
        {
            theElement.e_flags |= LOCATION_ELEMENT_SINGLE;
        }
            
        if (isComplement)
            readClosingParenthesis();

        theLocation.LocationElements.push_back(theElement);
    }

    void FeatureLocationParser::readBasePosition(long & a, long & b, unsigned & flags)
    {
        switch (peekToken())
        {
        case NUMBER:                 
            readNumber(a);
            flags = BASE_POSITION_SIMPLE_BASE;
            break;

        case OPENING_ANGLE_BRACKET:  
            readOpeningAngleBracket();
            readNumber(a);
            flags = BASE_POSITION_LEFT_OPEN;
            break;

        case CLOSING_ANGLE_BRACKET:  
            readClosingAngleBracket();
            readNumber(a);
            flags = BASE_POSITION_RIGHT_OPEN;
            break;

        case OPENING_PARENTHESIS:          
            readOpeningParenthesis();
            readNumber(a);
            readDot();
            readNumber(b);
            readClosingParenthesis();
            flags = BASE_POSITION_SPAN;
            break;

        default:
            badToken(NUMBER, getToken());
        }
    }


}
}
