/**++
 *   
 *   LICENSE
 *   -------
 *   
 *   Copyright (c) 2004 Renato Mancuso
 *   All rights reserved.
 *   
 *   Redistribution and use in source and binary forms, with or without modification, are 
 *   permitted provided that the following conditions are met:
 *   
 *   - Redistributions of source code must retain the above copyright notice, this list 
 *     of conditions and the following disclaimer.
 *   
 *   - Redistributions in binary form must reproduce the above copyright notice, this list
 *     of conditions and the following disclaimer in the documentation and/or other materials 
 *     provided with the distribution.
 *   
 *   - Neither the name of Renato Mancuso nor the names of its contributors may be used to 
 *     endorse or promote products derived from this software without specific prior written 
 *     permission.
 *   
 *   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS AS IS AND ANY EXPRESS 
 *   OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY 
 *   AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR 
 *   CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL 
 *   DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, 
 *   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER 
 *   IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT 
 *   OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *   
--**/


#ifndef OPEN_EMBL_FEATURE_LOCATION_PARSER_INCLUDED
#define OPEN_EMBL_FEATURE_LOCATION_PARSER_INCLUDED

#if !defined(OPEN_EMBL_NO_PRAGMA_ONCE)
#pragma once
#endif

#include <string>

#include "FieldDefs.hpp"
#include "ParserCtx.hpp"
#include "LocationLexer.hpp"

namespace OpenEMBL
{
namespace Phoenix
{
    
    class FeatureLocationParser
        : private virtual IParserCtx
    {
    public:
        FeatureLocationParser(IParserCtx* theParserCtx);
        bool parse(FeatureLocation & theLocation);

    private:
        // TextSource (ParserCtx)
        METHOD_(LPCSTR) currentLine() const NO_THROW;
        METHOD_(long)   currentLineNumber() const NO_THROW;
        METHOD_(bool)   nextLine();        

        // ErrorLog (ParserCtx)
        METHOD logError(long theLineNumber, int theErrorCode, LPCSTR theMessage);
        METHOD logWarning(long theLineNumber, int theWarning, LPCSTR theMessage);

    private:
        int         peekToken() const throw();
        int         getToken();        
        std::string getTokenValue() const;

        void badToken(int theExpectedToken, int theFoundToken);

        void readToken(int whichToken);
        void readToken(int whichToken, std::string & theValue);
        
        void readAccessionNumber(std::string & theValue);
        void readNumber(std::string & theValue);
        void readNumber(long & theValue);
        void readDoubleDot();
        void readJoin();
        void readOrder();
        void readComplement();
        void readComma();
        void readDot();
        void readColon();
        void readCaret();
        void readOpeningParenthesis();
        void readClosingParenthesis();
        void readOpeningAngleBracket();
        void readClosingAngleBracket();
        void readElement(FeatureLocation & theLocation);
        void readBasePosition(long & a, long & b, unsigned & flags);
        void parseLocation(FeatureLocation & theLocation);        

    private:
        struct InvalidLocation {};

        // Numerical codes of tokens (EOF is returned for end of file)
        enum
        {
            BAD_TOKEN               = LocationLexer::BAD_TOKEN,
            ACCESSION_NUMBER        = LocationLexer::ACCESSION_NUMBER,    
            NUMBER                  = LocationLexer::NUMBER,    
            DOUBLE_DOT              = LocationLexer::DOUBLE_DOT,    
            JOIN                    = LocationLexer::JOIN,
            ORDER                   = LocationLexer::ORDER,
            COMPLEMENT              = LocationLexer::COMPLEMENT,
            GAP                     = LocationLexer::GAP,
            COMMA                   = LocationLexer::COMMA,
            DOT                     = LocationLexer::DOT, 
            COLON                   = LocationLexer::COLON,
            CARET                   = LocationLexer::CARET, 
            OPENING_PARENTHESIS     = LocationLexer::OPENING_PARENTHESIS,
            CLOSING_PARENTHESIS     = LocationLexer::CLOSING_PARENTHESIS,
            OPENING_ANGLE_BRACKET   = LocationLexer::OPENING_ANGLE_BRACKET,
            CLOSING_ANGLE_BRACKET   = LocationLexer::CLOSING_ANGLE_BRACKET
        };

        static std::string toString(int theToken);

    private:
        IParserCtx*         m_ParserCtx;
        LocationLexer       m_Lexer;
    };

}
}

#endif // OPEN_EMBL_FEATURE_LOCATION_PARSER_INCLUDED
