/**++
 *   
 *   LICENSE
 *   -------
 *   
 *   Copyright (c) 2004 Renato Mancuso
 *   All rights reserved.
 *   
 *   Redistribution and use in source and binary forms, with or without modification, are 
 *   permitted provided that the following conditions are met:
 *   
 *   - Redistributions of source code must retain the above copyright notice, this list 
 *     of conditions and the following disclaimer.
 *   
 *   - Redistributions in binary form must reproduce the above copyright notice, this list
 *     of conditions and the following disclaimer in the documentation and/or other materials 
 *     provided with the distribution.
 *   
 *   - Neither the name of Renato Mancuso nor the names of its contributors may be used to 
 *     endorse or promote products derived from this software without specific prior written 
 *     permission.
 *   
 *   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS AS IS AND ANY EXPRESS 
 *   OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY 
 *   AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR 
 *   CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL 
 *   DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, 
 *   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER 
 *   IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT 
 *   OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *   
--**/


#include "FeatureQualifierParser.hpp"
#include "ParserErrors.hpp"
#include "ParserWarnings.hpp"
#include "StringUtils.hpp"
#include "Regex.hpp"

#include <cctype>
#include <cstring>
#include <cassert>

namespace OpenEMBL
{
namespace Phoenix
{

    FeatureQualifierParser::FeatureQualifierParser(
        IParserCtx*                      theParserCtx,
        IItemHandler<FeatureQualifier>*  theHandler)
        : ItemParserImpl<FeatureQualifier>(theParserCtx, theHandler)
    {
    }

    METHODIMP FeatureQualifierParser::parse()
    {
        long   theLineNumber = currentLineNumber();
        LPCSTR theLine       = currentLine();

        if (NULL == theLine)
        {
            logError(theLineNumber,
                     ERROR_EOF_FOUND,
                     NULL);
            return;
        }

        if (!isQualifierLine(theLine))
        {
            logError(theLineNumber,
                     ERROR_INVALID_FEATURE_QUALIFIER_LINE,
                     theLine);
            nextLine();
            return;
        }

        FeatureQualifier theItem;
        
        if (!extractLines(theItem.Name, theItem.Value))
            return;
        
        notifyParsed(theLineNumber, theItem);
    }

    bool FeatureQualifierParser::isQualifierLine(char const * theLine)
    {
        assert(NULL != theLine);
        
        std::string theData = trimLeft(theLine);

        if (theData.length() < 2)
            return false;

        return theData[0] == '/';
    }

    bool FeatureQualifierParser::extractLines(
                        String &        theQualifier, 
                        StringList &    theQualifierValue)
    {
        //
        // Due to the rather stupid definition of the format 
        // we are forced to parse as we read the lines :(
        //
        long    theLineNumber   = currentLineNumber();        
        String  theLine         = currentLine();
        nextLine();

        assert(theLine.length() >= 2); // see definition of isQualifierLine()

        if (theLine[0] != '/')
        {
            logWarning(theLineNumber,
                       WARNING_FEATURE_QUAL_NOT_IN_PLACE,
                       theLine.c_str());

            theLine = trimLeft(theLine);
        }

        assert(theLine[0] == '/'); // see definition of isQualifierLine()

        theLine.erase(0, 1);

        if (isspace(theLine[0]))
        {
            logWarning(theLineNumber,
                       WARNING_SPACE_BEFORE_QUALIFIER_NAME,
                       NULL);

            theLine = trimLeft(theLine);
        }        

        if (theLine.empty())
        {
            logError(theLineNumber,
                     ERROR_QUALIFIER_NOT_FOUND,
                     NULL);
            return false;
        }       

        String::size_type thePosition = theLine.find('=');

        if (std::string::npos == thePosition)
        {
            theQualifier = trimRight(theLine);
            return true;
        }      

        theQualifier = theLine.substr(0, thePosition);

        if (theQualifier != trim(theQualifier))
        {
            logWarning(theLineNumber,
                       WARNING_SPACE_AFTER_QUALIFIER_NAME,
                       NULL);

            theQualifier = trim(theQualifier);
        }        

        theLine.erase(0, thePosition + 1);
        
        if (theLine.length() > 0 && isspace(theLine[0]))
        {
            logWarning(theLineNumber,
                       WARNING_SPACE_BEFORE_QUALIFIER_VALUE,
                       NULL);

            theLine = trimLeft(theLine);
        }        

        if (theLine.empty())
        {
            logError(theLineNumber,
                     ERROR_MISSING_QUALIFIER_VALUE,
                     NULL);
            return false;
        }                

        if (theLine[0] == '"') 
        {
            // quoted text - can span multiple lines
            theLine = trimRight(theLine);                     
            theQualifierValue.push_back(theLine);

            if (!isQualifierLastLine(theLine.erase(0, 1)))
            {
                while (1)
                {                   
                    if (NULL == currentLine())
                    {
                        logError(currentLineNumber(),
                                 ERROR_QUALIFIER_VALUE_MISSING_TERMINAL_QUOTES,
                                 NULL);

                        return false;
                    }

                    std::string s = trimRight(currentLine());
                    nextLine();

                    theQualifierValue.push_back(s);                                                 

                    if (isQualifierLastLine(s))
                        break;
                }
            }
        }
        else
        {
            // unquoted value - can span only 1 line
            theLine = trimRight(theLine);         
            theQualifierValue.push_back(theLine);
        }

        return true;
    }

    bool FeatureQualifierParser::isQualifierLastLine(std::string const & theLine)
    {
        enum STATE { INQUOTE, OUTQUOTE };

        STATE state = OUTQUOTE;

        for (size_t i = 0; i < theLine.length(); ++i)
        {
            char c = theLine[i];

            switch (state)
            {
            case INQUOTE:
                state = OUTQUOTE;
                break;

            case OUTQUOTE:
                if (c == '"') state = INQUOTE;
                break;
            }
        }

        return state == INQUOTE;
    }

}
}
