/**++
 *   
 *   LICENSE
 *   -------
 *   
 *   Copyright (c) 2004 Renato Mancuso
 *   All rights reserved.
 *   
 *   Redistribution and use in source and binary forms, with or without modification, are 
 *   permitted provided that the following conditions are met:
 *   
 *   - Redistributions of source code must retain the above copyright notice, this list 
 *     of conditions and the following disclaimer.
 *   
 *   - Redistributions in binary form must reproduce the above copyright notice, this list
 *     of conditions and the following disclaimer in the documentation and/or other materials 
 *     provided with the distribution.
 *   
 *   - Neither the name of Renato Mancuso nor the names of its contributors may be used to 
 *     endorse or promote products derived from this software without specific prior written 
 *     permission.
 *   
 *   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS AS IS AND ANY EXPRESS 
 *   OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY 
 *   AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR 
 *   CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL 
 *   DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, 
 *   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER 
 *   IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT 
 *   OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *   
--**/


#include "FeatureTableParser.hpp"
#include "ParserErrors.hpp"
#include "ParserWarnings.hpp"
#include "FeatureKeyParser.hpp"
#include "StringUtils.hpp"

#include <algorithm>
#include <cstring>
#include <cassert>

namespace OpenEMBL
{
namespace Phoenix
{

    FeatureTableParser::FeatureTableParser(
        IParserCtx*             pParserCtx,
        IFeatureTableHandler*   pFeatureTableHandler,
        FormatType              theFormat )
        : m_pParserCtx(pParserCtx)
        , m_pFeatureTableHandler(pFeatureTableHandler)
        , m_TextSource(pParserCtx, 
                       theFormat == EMBL ? 
                       FeatureTableTextSourceAdapter::EMBL :
                       FeatureTableTextSourceAdapter::GENBANK)
    {
    }

    METHODIMP_(LPCSTR) FeatureTableParser::currentLine() const NO_THROW
    {
        return m_TextSource.currentLine();
    }

    METHODIMP_(long) FeatureTableParser::currentLineNumber() const NO_THROW
    {
        return m_TextSource.currentLineNumber();    
    }

    METHODIMP_(bool) FeatureTableParser::nextLine()
    {
        return m_TextSource.nextLine();
    }

    METHODIMP FeatureTableParser::logError(long theLineNumber, int theErrorCode, char const * theMessage)
    {
        if (NULL != m_pParserCtx)
            m_pParserCtx->logError(theLineNumber, theErrorCode, theMessage);
    }

    METHODIMP FeatureTableParser::logWarning(long theLineNumber, int theWarning, char const * theMessage)
    {
        if (NULL != m_pParserCtx)
            m_pParserCtx->logWarning(theLineNumber, theWarning, theMessage);
    }

    void FeatureTableParser::notifyBeginFeatureTable(long theLineNumber)
    {
        if (NULL != m_pFeatureTableHandler)
            m_pFeatureTableHandler->onBeginFeatureTable(theLineNumber);
    }

    void FeatureTableParser::notifyEndFeatureTable()
    {
        if (NULL != m_pFeatureTableHandler)
            m_pFeatureTableHandler->onEndFeatureTable();
    }

    bool FeatureTableParser::isFeatureKeyLine(char const * theLine)
    {
        if (NULL == theLine)
            return false;

        std::string theRow = theLine;

        if (theRow.length() < 6)
            return false;

        std::string theKey = trimLeft(theRow.substr(5, 15));

        return !theKey.empty();
    }

    void FeatureTableParser::goToNextKey()
    {
        while (1)
        {
            if (NULL == currentLine())
                break;

            if (isFeatureKeyLine(currentLine()))
                break;

            nextLine();
        }
    }

    METHODIMP FeatureTableParser::parse()
    {
        if (!isFeatureKeyLine(currentLine()))
        {
            logError(currentLineNumber(),
                     ERROR_MISSING_FEATURE_KEY,
                     m_pParserCtx ? m_pParserCtx->currentLine() : NULL);

            logWarning(currentLineNumber(),
                       WARNING_SKIPPING_LINES_SEARCHING_FOR_FKEY,
                       NULL);

            // by contract we always need to advance the outer text source
            // in order to consume at least the line that the user
            // passed in
            if (NULL != m_pParserCtx)
            {
                m_pParserCtx->nextLine();
                m_TextSource.resynch();
            }

            goToNextKey();
        }

        if (NULL == currentLine())
            return;

        notifyBeginFeatureTable(currentLineNumber());

        while (NULL != currentLine())
            parseKey();

        notifyEndFeatureTable();            
    }   

    void FeatureTableParser::parseKey()
    {
        FeatureKeyParser theParser(this, m_pFeatureTableHandler);
        theParser.parse();        
    }

}
}
