/**++
 *   
 *   LICENSE
 *   -------
 *   
 *   Copyright (c) 2004 Renato Mancuso
 *   All rights reserved.
 *   
 *   Redistribution and use in source and binary forms, with or without modification, are 
 *   permitted provided that the following conditions are met:
 *   
 *   - Redistributions of source code must retain the above copyright notice, this list 
 *     of conditions and the following disclaimer.
 *   
 *   - Redistributions in binary form must reproduce the above copyright notice, this list
 *     of conditions and the following disclaimer in the documentation and/or other materials 
 *     provided with the distribution.
 *   
 *   - Neither the name of Renato Mancuso nor the names of its contributors may be used to 
 *     endorse or promote products derived from this software without specific prior written 
 *     permission.
 *   
 *   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS AS IS AND ANY EXPRESS 
 *   OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY 
 *   AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR 
 *   CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL 
 *   DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, 
 *   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER 
 *   IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT 
 *   OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *   
--**/


#ifndef OPEN_EMBL_HANDLER_DISPATCH_INCLUDED
#define OPEN_EMBL_HANDLER_DISPATCH_INCLUDED

#if !defined(OPEN_EMBL_NO_PRAGMA_ONCE)
#pragma once
#endif

#include <map>
#include <string>
#include <sstream>
#include <cstring>
#include <cassert>

#include "NonCopyable.hpp"
#include "ParserErrors.hpp"
#include "HandlerLookup.hpp"

namespace OpenEMBL
{
namespace Phoenix
{

    template <class T>
    class HandlerDispatch
        : public HandlerLookup<T>
        , NonCopyable
    {
    public:
        typedef typename HandlerLookup<T>::HANDLER       HANDLER;
        typedef typename HandlerLookup<T>::HANDLER_REC   HANDLER_REC;

        HandlerDispatch();
        ~HandlerDispatch() throw();

        void setTargetInstance         (T* theInstance);
        void registerStartHandler      (HANDLER theHandler);
        void registerTerminationHandler(HANDLER theHandler);
        void setUsageLimit(std::string const & theFieldCode, size_t theLimit);
        void goToNextRecord();
        void resetCounters();

        bool dispatch();

    private:
        bool         isStartOfRecord (char const* theLine) const;
        bool         isEndOfRecord   (char const* theLine) const;
        char const*  getStartOfRecordHeader() const;
        bool         ignoreField(char const* theFieldCode) const;

        char const*  currentLine() const throw();
        long         currentLineNumber() const throw();
        bool         nextLine();

        void logError  (long theLineNumber, int theErrorCode, char const* theMessage);
        void logWarning(long theLineNumber, int theWarningNumber, char const* theMessage);

        void invoke(HANDLER theHandler);

    private:
        T*                              m_Instance;
        HANDLER                         m_StartHandler;
        HANDLER                         m_TerminationHandler;
        std::map<std::string, int>      m_Counters;
        std::map<std::string, int>      m_UsageMap;
    };

    template <class T>
        HandlerDispatch<T>::HandlerDispatch()
        : m_Instance(NULL)
        , m_StartHandler(NULL)
        , m_TerminationHandler(NULL)
    {        
    }

    template <class T>
        HandlerDispatch<T>::~HandlerDispatch() throw()
    {        
    }

    template <class T>
        bool HandlerDispatch<T>::isStartOfRecord (char const* theLine) const
    {
        assert(NULL != theLine);
        return m_Instance->isStartOfRecord(theLine);
    }

    template <class T>
        bool HandlerDispatch<T>::isEndOfRecord (char const* theLine) const
    {
        assert(NULL != theLine);
        return m_Instance->isEndOfRecord(theLine);
    }

    template <class T>
        char const* HandlerDispatch<T>::getStartOfRecordHeader() const
    {
        return m_Instance->getStartOfRecordHeader();
    }

    template <class T>
        bool HandlerDispatch<T>::ignoreField(char const* theFieldCode) const
    {
        return m_Instance->ignoreField(theFieldCode);
    }

    template <class T>
        char const* HandlerDispatch<T>::currentLine() const throw()
    {
        return m_Instance->currentLine();
    }

    template <class T>
        long HandlerDispatch<T>::currentLineNumber() const throw()
    {
        return m_Instance->currentLineNumber();
    }

    template <class T>
        bool HandlerDispatch<T>::nextLine()
    {
        return m_Instance->nextLine();
    }

    template <class T>
        void HandlerDispatch<T>::logError (
        long            theLineNumber, 
        int             theErrorCode, 
        char const*     theMessage)
    {
        m_Instance->logError(theLineNumber, theErrorCode, theMessage);
    }

    template <class T>
        void HandlerDispatch<T>::logWarning(
        long            theLineNumber, 
        int             theWarningNumber, 
        char const*    theMessage)
    {
        m_Instance->logWarning(theLineNumber, theWarningNumber, theMessage);
    }

    template <class T>
        void HandlerDispatch<T>::invoke(HANDLER theHandler)
    {
        if (NULL != theHandler)
            (m_Instance->*theHandler)();
    }

    template <class T>
        void HandlerDispatch<T>::setTargetInstance(T* theInstance)
    {
        assert(NULL != theInstance);
        m_Instance = theInstance;
    }

    template <class T>
        void HandlerDispatch<T>::registerStartHandler(HANDLER theHandler)
    {
        assert(NULL != theHandler);
        m_StartHandler = theHandler;
    }    

    template <class T>
        void HandlerDispatch<T>::registerTerminationHandler(HANDLER theHandler)
    {
        assert(NULL != theHandler);
        m_TerminationHandler = theHandler;
    }    

    template <class T>
        void HandlerDispatch<T>::goToNextRecord()
    {
        for (;;)
        {
            if (NULL == currentLine())
                break;

            if (isStartOfRecord(currentLine()))
                break;

            nextLine();
        }
    }

    template <class T>
        void HandlerDispatch<T>::resetCounters()
    {
        m_Counters.clear();
    }

    template <class T>
        void HandlerDispatch<T>::setUsageLimit(std::string const & theFieldCode, size_t theLimit /* = 1*/)
    {
        m_UsageMap[theFieldCode] = (int) theLimit;
    }    

    template <class T>
        bool HandlerDispatch<T>::dispatch()
    {
        assert(NULL != m_Instance);
        assert(NULL != m_StartHandler);
        assert(NULL != m_TerminationHandler);

        resetCounters();

        long theLineNumber = currentLineNumber();

        if (NULL == currentLine())
        {
            logError(theLineNumber,
                     ERROR_EOF_FOUND,
                     NULL);
            return false;
        }            

        if (false == isStartOfRecord(currentLine()))
        {
            std::ostringstream oss;

            oss << "expected line starting with '"
                << getStartOfRecordHeader()
                << "', found instead '"
                << currentLine()
                << "'"
                ;

            logError(theLineNumber,
                     ERROR_INVALID_START_OF_RECORD,
                     oss.str().c_str());
            return false;
        }

        invoke(m_StartHandler);

        for (;;)
        {            
            char const* theLine = currentLine();

            theLineNumber = currentLineNumber();

            if (NULL == theLine)                
                break;

            if (isEndOfRecord(theLine))
                break;

            HANDLER_REC theHandlerRec;

            if (!lookupHandler(theLine, &theHandlerRec))
            {
                logError(theLineNumber,
                         ERROR_UNKNOWN_LINE_TYPE,
                         theLine);
                nextLine();
                continue;
            }

            if (ignoreField(theHandlerRec.FieldCode.c_str()))
            {
                nextLine();
                continue;
            }

            m_Counters[theHandlerRec.FieldCode] = m_Counters[theHandlerRec.FieldCode] + 1;

            if (m_UsageMap[theHandlerRec.FieldCode] > 0)
            {
                if (m_Counters[theHandlerRec.FieldCode] > m_UsageMap[theHandlerRec.FieldCode])
                {
                    logError(theLineNumber,
                             ERROR_DUPLICATE_LINE,
                             theLine); 
                    nextLine();
                    continue;
                }            
            }

            invoke(theHandlerRec.Handler);
            assert(theLineNumber <currentLineNumber());

            if (theLineNumber == currentLineNumber())
            {
                logError(theLineNumber,
                         ERROR_INTERNAL_ERROR_INFINITE_LOOP,
                         "INFINITE LOOP DETECTED");
                nextLine();
            }
        }

        invoke(m_TerminationHandler);
        return true;
    }

}
}

#endif // OPEN_EMBL_HANDLER_DISPATCH_INCLUDED
