/**++
 *   
 *   LICENSE
 *   -------
 *   
 *   Copyright (c) 2004 Renato Mancuso
 *   All rights reserved.
 *   
 *   Redistribution and use in source and binary forms, with or without modification, are 
 *   permitted provided that the following conditions are met:
 *   
 *   - Redistributions of source code must retain the above copyright notice, this list 
 *     of conditions and the following disclaimer.
 *   
 *   - Redistributions in binary form must reproduce the above copyright notice, this list
 *     of conditions and the following disclaimer in the documentation and/or other materials 
 *     provided with the distribution.
 *   
 *   - Neither the name of Renato Mancuso nor the names of its contributors may be used to 
 *     endorse or promote products derived from this software without specific prior written 
 *     permission.
 *   
 *   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS AS IS AND ANY EXPRESS 
 *   OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY 
 *   AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR 
 *   CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL 
 *   DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, 
 *   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER 
 *   IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT 
 *   OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *   
--**/


#ifndef OPEN_EMBL_HANDLER_LOOKUP_INCLUDED
#define OPEN_EMBL_HANDLER_LOOKUP_INCLUDED

#if !defined(OPEN_EMBL_NO_PRAGMA_ONCE)
#pragma once
#endif

#include <vector>
#include <string>
#include <cstring>

namespace OpenEMBL
{
namespace Phoenix
{

    template <class T>
    class HandlerLookup
    {
    public:
        typedef void (T::*HANDLER)(void);

        struct HANDLER_REC
        {
            std::string     Header;
            std::string     FieldCode;
            HANDLER         Handler;

            HANDLER_REC()
                : Handler(NULL)
            {
            }

            HANDLER_REC(
                std::string const & theHeader, 
                std::string const & theFieldCode, 
                HANDLER             theHandler)
                : Header    (theHeader)
                , FieldCode (theFieldCode)
                , Handler   (theHandler)
            {
            }
        };

        virtual void registerHandler(
            std::string const & theHeader,    
            std::string const & theFieldCode, 
            HANDLER             theHandler); 

        virtual bool lookupHandler(char const* theLine, HANDLER_REC* theHandlerRec);

    protected:
        virtual ~HandlerLookup() { }

    private:
        std::vector<HANDLER_REC>  m_Handlers;
    };

    template <class T>
        void HandlerLookup<T>::registerHandler(
        std::string const & theHeader, 
        std::string const & theFieldCode,
        HANDLER             theHandler)
    {
        if (NULL != theHandler)
            m_Handlers.push_back(HANDLER_REC(theHeader, theFieldCode, theHandler));
    }

    template <class T>
        bool HandlerLookup<T>::lookupHandler(
        char const*    theLine, 
        HANDLER_REC*   theHandlerRec)
    {
        if (NULL == theLine)
            return false;

        for (size_t i = 0; i <m_Handlers.size(); ++i)
        {
            HANDLER_REC & theItem = m_Handlers[i];

            if (0 == strncmp(theLine, theItem.Header.c_str(), theItem.Header.length()))
            {
                if (NULL != theHandlerRec)
                    *theHandlerRec = theItem;

                return true;
            }
        }

        return false;
    }

}
}

#endif // OPEN_EMBL_HANDLER_LOOKUP_INCLUDED
