/**++
 *   
 *   LICENSE
 *   -------
 *   
 *   Copyright (c) 2004 Renato Mancuso
 *   All rights reserved.
 *   
 *   Redistribution and use in source and binary forms, with or without modification, are 
 *   permitted provided that the following conditions are met:
 *   
 *   - Redistributions of source code must retain the above copyright notice, this list 
 *     of conditions and the following disclaimer.
 *   
 *   - Redistributions in binary form must reproduce the above copyright notice, this list
 *     of conditions and the following disclaimer in the documentation and/or other materials 
 *     provided with the distribution.
 *   
 *   - Neither the name of Renato Mancuso nor the names of its contributors may be used to 
 *     endorse or promote products derived from this software without specific prior written 
 *     permission.
 *   
 *   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS AS IS AND ANY EXPRESS 
 *   OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY 
 *   AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR 
 *   CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL 
 *   DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, 
 *   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER 
 *   IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT 
 *   OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *   
--**/


#ifndef OPEN_EMBL_ITEM_PARSER_IMPL_INCLUDED
#define OPEN_EMBL_ITEM_PARSER_IMPL_INCLUDED

#if !defined(OPEN_EMBL_NO_PRAGMA_ONCE)
#   pragma once
#endif

#include <vector>

#include "ItemParser.hpp"
#include "ParserCtx.hpp"
#include "ItemHandler.hpp"

namespace OpenEMBL
{
namespace Phoenix
{

    template <class ITEM>
    class ItemParserImpl
        : public    virtual IItemParser<ITEM>
        , protected virtual IParserCtx
    {
    public:
        ItemParserImpl(
            IParserCtx*             pParserCtx,
            IItemHandler<ITEM>*     pItemHandler);

    protected:
        // TextSource (ParserCtx)
        METHOD_(LPCSTR) currentLine() const NO_THROW;
        METHOD_(long)   currentLineNumber() const NO_THROW;
        METHOD_(bool)   nextLine();

        // ErrorLog (ParserCtx)
        METHOD logError(long theLineNumber, int theErrorCode, LPCSTR theMessage);
        METHOD logWarning(long theLineNumber, int theWarning, LPCSTR theMessage);

        // ItemParser<ITEM>
        METHOD notifyParsed(long theLineNumber, ITEM const & theItem);

        // utility routine to extract all lines belonging to a field
        template <class PRED>
            void extractLines(PRED pred, std::vector<std::string> & dest)
        {
            std::vector<std::string> tmp;

            while (1)
            {
                if (NULL == currentLine())
                    break;

                if (false == pred(currentLine()))
                    break;

                tmp.push_back(currentLine());
                nextLine();
            }

            tmp.swap(dest);
        }

    private:
        IParserCtx*             m_pParserCtx;
        IItemHandler<ITEM>*     m_pItemHandler;
    };

    template <class ITEM>
    ItemParserImpl<ITEM>::ItemParserImpl(
        IParserCtx*             pParserCtx, 
        IItemHandler<ITEM>*     pItemHandler)
        : m_pParserCtx(pParserCtx)
        , m_pItemHandler(pItemHandler)
    {
    }

    template <class ITEM>
    METHODIMP_(LPCSTR) ItemParserImpl<ITEM>::currentLine() const NO_THROW
    {
        return m_pParserCtx != NULL ? m_pParserCtx->currentLine() : NULL;
    }

    template <class ITEM>
    METHODIMP_(long) ItemParserImpl<ITEM>::currentLineNumber() const NO_THROW
    {
        return m_pParserCtx != NULL ? m_pParserCtx->currentLineNumber() : TEXTSOURCE_INVALID_LINE_NUMBER;
    }

    template <class ITEM>
    METHODIMP_(bool) ItemParserImpl<ITEM>::nextLine()
    {
        return m_pParserCtx != NULL ? m_pParserCtx->nextLine() : false;
    }

    template <class ITEM>
    METHODIMP ItemParserImpl<ITEM>::logError(long theLineNumber, int theErrorCode, LPCSTR theMessage)
    {
        if (NULL != m_pParserCtx)
            m_pParserCtx->logError(theLineNumber, theErrorCode, theMessage);
    }

    template <class ITEM>
    METHODIMP ItemParserImpl<ITEM>::logWarning(long theLineNumber, int theWarning, LPCSTR theMessage)
    {
        if (NULL != m_pParserCtx)
            m_pParserCtx->logWarning(theLineNumber, theWarning, theMessage);
    }

    template <class ITEM>
    METHODIMP ItemParserImpl<ITEM>::notifyParsed(long theLineNumber, ITEM const & theItem)
    {
        if (NULL != m_pItemHandler)
            m_pItemHandler->onParsed(theLineNumber, theItem);
    }

}
}

#endif // OPEN_EMBL_ITEM_PARSER_IMPL_INCLUDED
