/**++
 *   
 *   LICENSE
 *   -------
 *   
 *   Copyright (c) 2004 Renato Mancuso
 *   All rights reserved.
 *   
 *   Redistribution and use in source and binary forms, with or without modification, are 
 *   permitted provided that the following conditions are met:
 *   
 *   - Redistributions of source code must retain the above copyright notice, this list 
 *     of conditions and the following disclaimer.
 *   
 *   - Redistributions in binary form must reproduce the above copyright notice, this list
 *     of conditions and the following disclaimer in the documentation and/or other materials 
 *     provided with the distribution.
 *   
 *   - Neither the name of Renato Mancuso nor the names of its contributors may be used to 
 *     endorse or promote products derived from this software without specific prior written 
 *     permission.
 *   
 *   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS AS IS AND ANY EXPRESS 
 *   OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY 
 *   AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR 
 *   CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL 
 *   DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, 
 *   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER 
 *   IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT 
 *   OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *   
--**/


#include "ListReader.hpp"
#include "StringUtils.hpp"
#include "ParserWarnings.hpp"

#include <cctype>
#include <sstream>
#include <algorithm>
#include <vector>
#include <string>
#include <cassert>

namespace OpenEMBL
{
namespace Phoenix
{

    ListReader::ListReader(
        IErrorLog*  theErrorLog, 
        char        theListSeparator, 
        char        theListTerminator,
        unsigned    theOptions)
        : m_ErrorLog(theErrorLog)
        , m_ListSeparator(theListSeparator)
        , m_ListTerminator(theListTerminator)
        , m_Options(theOptions)
    {
    }

    METHODIMP ListReader::logError(long theLineNumber, int theErrorCode, LPCSTR theMessage)
    {
        if (NULL != m_ErrorLog)
            m_ErrorLog->logError(theLineNumber, theErrorCode, theMessage);
    }

    METHODIMP ListReader::logWarning(long theLineNumber, int theWarning, LPCSTR theMessage)
    {
        if (NULL != m_ErrorLog)
            m_ErrorLog->logWarning(theLineNumber, theWarning, theMessage);
    }

    void ListReader::readList(
            long                                theLineNumber,
            std::string const &                 theLine,
            std::vector<std::string>       &    theListElements)
    {
        std::vector<std::string> theLines;
        theLines.push_back(theLine);
        readList(theLineNumber, theLines, theListElements);
    }

    void ListReader::readList(
            long                                theLineNumber,
            std::vector<std::string> const &    theLines,
            std::vector<std::string>       &    theListElements)
    {
        std::vector<std::string> tmp;

        if (theLines.size() > 0)
        {
            for (size_t i = 0; i < theLines.size() - 1; ++i)
                processNonTerminalLine(trimRight(theLines[i]), theLineNumber + long(i), tmp);

            processTerminalLine(
                trimRight(theLines[theLines.size() - 1]), 
                theLineNumber + long(theLines.size() - 1),   
                tmp );
        }

        std::transform(tmp.begin(), 
                       tmp.end(),
                       tmp.begin(),
                       trim);

        tmp.swap(theListElements);
    }

    void ListReader::raiseWarning(
            long    theLineNumber,
            int     theWarning,
            char    theExpectedChar,
            char    theActualChar )
    {
            std::ostringstream oss;

            oss << "expected '" << theExpectedChar 
                << "', found '";

            if (0 == theActualChar)
                oss << "<NULL>";
            else if (!isprint(theActualChar))
                oss << "#" << int(theActualChar);
            else
                oss << theActualChar;

            oss << "'";

            logWarning(theLineNumber,
                       theWarning,
                       oss.str().c_str());
    }

    void ListReader::extractElements(
            std::string::const_iterator first,
            std::string::const_iterator last,
            std::vector<std::string> &  theListElements )
    {
        typedef std::string::const_iterator iterator;

        iterator element_begin = first;

        while (element_begin != last)
        {
            iterator element_end = std::find(element_begin, last, m_ListSeparator);
            theListElements.push_back(std::string(element_begin, element_end));
            if (element_end == last) break;       
            element_begin = element_end + 1;            
        }
    }
 
    void ListReader::processNonTerminalLine(
            std::string const &         theLine,
            long                        theLineNumber,
            std::vector<std::string> &  theListElements )
    {
        typedef std::string::const_iterator iterator;

        iterator first = theLine.begin();
        iterator last  = theLine.end();

        if (first == last || *(last - 1) != m_ListSeparator)
        {
            raiseWarning(
                theLineNumber,
                WARNING_MISSING_LIST_SEPARATOR,
                m_ListSeparator,
                (first == last)? 0 : *(last - 1));
        }
        else
        {
            // remove terminator
            --last;
        }

        extractElements(first, last, theListElements);
    }

    void ListReader::processTerminalLine(
            std::string const &         theLine,
            long                        theLineNumber,
            std::vector<std::string> &  theListElements )
    {
        typedef std::string::const_iterator iterator;

        iterator first = theLine.begin();
        iterator last  = theLine.end();

        if (m_Options & HAS_TERMINATOR)
        {
            if (first == last || *(last - 1) != m_ListTerminator)
            {
                raiseWarning(
                    theLineNumber,
                    WARNING_MISSING_LIST_TERMINATOR,
                    m_ListTerminator,
                    (first == last)? 0 : *(last - 1));
            }
            else
            {
                --last;
            }
        }

        extractElements(first, last, theListElements);
    }

}
}
