/**++
 *   
 *   LICENSE
 *   -------
 *   
 *   Copyright (c) 2004 Renato Mancuso
 *   All rights reserved.
 *   
 *   Redistribution and use in source and binary forms, with or without modification, are 
 *   permitted provided that the following conditions are met:
 *   
 *   - Redistributions of source code must retain the above copyright notice, this list 
 *     of conditions and the following disclaimer.
 *   
 *   - Redistributions in binary form must reproduce the above copyright notice, this list
 *     of conditions and the following disclaimer in the documentation and/or other materials 
 *     provided with the distribution.
 *   
 *   - Neither the name of Renato Mancuso nor the names of its contributors may be used to 
 *     endorse or promote products derived from this software without specific prior written 
 *     permission.
 *   
 *   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS AS IS AND ANY EXPRESS 
 *   OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY 
 *   AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR 
 *   CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL 
 *   DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, 
 *   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER 
 *   IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT 
 *   OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *   
--**/


#include "LocationLexer.hpp"
#include "ParserErrors.hpp"
#include "StringUtils.hpp"

#include <cstdlib>
#include <cstdio>
#include <cstring>
#include <sstream>
#include <cassert>

namespace OpenEMBL
{
namespace Phoenix
{

    LocationLexer::LocationLexer()
        : m_ErrorLog(NULL)
        , m_CurrentToken(BAD_TOKEN, '@')
        , m_NextToken(BAD_TOKEN, '@')
    {        
    }

    void 
    LocationLexer::setErrorLog(IErrorLog* theErrorLog)
    {
        m_ErrorLog = theErrorLog;
    }

    void 
    LocationLexer::setTextSource(ITextSource* theTextSource)
    {
        m_CharIterator.setTextSource(theTextSource);
        readToken();
    }

    METHODIMP_(int)    
    LocationLexer::peekToken() const NO_THROW
    {
        return m_NextToken.getCode();
    }

    METHODIMP_(int)    
    LocationLexer::getToken()
    {
        m_CurrentToken = m_NextToken;   

        readToken();        

        return m_CurrentToken.getCode();
    }

    METHODIMP_(bool)   
    LocationLexer::getTokenValue(std::string & theValue) const
    {
        if (!m_CurrentToken.isNull())
        {
            theValue = m_CurrentToken.getValue();    
            return true;
        }        

        return false;
    }

    void 
    LocationLexer::readToken()
    {
        skipBlanks();

        switch(currentChar())
        {
        case EOF:
            m_NextToken.assign(EOF, "");
            break;

        case '0':
        case '1': case '2': case '3':
        case '4': case '5': case '6':
        case '7': case '8': case '9': 
            readNumber();
            break;

        case '<': case '>': case '(': case ')':
        case ',': case '^': case ':': case '.':
            readPunct();
            break;
            
        default:
            if (isalpha(currentChar()))
            {
                readIdent();
            }
            else
            {
                std::ostringstream oss;

                oss << "expected one of <>()^:.,1-9A-Za-z, found instead "
                    << makePrintable((char) currentChar())                    
                    ;

                badToken(oss.str());

                nextChar();
            }
        }
    }

    int
    LocationLexer::currentChar()
    {
        return m_CharIterator.currentChar();
    }

    void
    LocationLexer::nextChar()
    {
        m_CharIterator.nextChar();
    }

    void 
    LocationLexer::skipBlanks()
    {
        while (isspace(currentChar()))
            nextChar();
    }

    void
    LocationLexer::readPunct()
    {
        switch(currentChar())
        {
        case '<':
            m_NextToken.assign(OPENING_ANGLE_BRACKET, '<');
            nextChar();
            break;

        case '>':
            m_NextToken.assign(CLOSING_ANGLE_BRACKET, '>');
            nextChar();
            break;

        case '(':
            m_NextToken.assign(OPENING_PARENTHESIS, '(');
            nextChar();
            break;

        case ')':
            m_NextToken.assign(CLOSING_PARENTHESIS, ')');
            nextChar();
            break;

        case '^':
            m_NextToken.assign(CARET, '^');
            nextChar();
            break;

        case ',':
            m_NextToken.assign(COMMA, ',');
            nextChar();
            break;

        case ':':
            m_NextToken.assign(COLON, ':');
            nextChar();
            break;

        case '.':
            nextChar();

            if ('.' == currentChar())
            {
                m_NextToken.assign(DOUBLE_DOT, "..");
                nextChar();
            }
            else
            {
                m_NextToken.assign(DOT, '.');
            }
            break;

        default:            
            // if you are here then you are calling readPunct()
            // when the currentChar() is not a recognized
            // character (ex: there is a mismatch between the
            // switch conditions here and the ones in readToken())
            // Make sure that the set of punctuation character
            // is the same in both routines

            assert(0);

            {
                std::ostringstream oss;

                oss << "expected one of <>(),^:., found instead "
                    << makePrintable((char) currentChar())                    
                    ;
                   
                badToken(oss.str());

                nextChar();
            }            
        } 
    }

    void
    LocationLexer::readNumber()
    {
        assert(isdigit(currentChar()));

        std::ostringstream oss;

        while (isdigit(currentChar()))
        {
            oss << (char) currentChar();
            nextChar();
        }

        m_NextToken.assign(NUMBER, oss.str());
    }

    void
    LocationLexer::readIdent()
    {
        assert(isalpha(currentChar()));

        std::ostringstream oss;

        while (isalnum(currentChar()))
        {
            oss << (char) currentChar();
            nextChar();
        }

        std::string theValue = oss.str();

        if (0 == strCaseCmp(theValue.c_str(), "complement"))
        {
            m_NextToken.assign(COMPLEMENT, "complement");
        }
        else if (0 == strCaseCmp(theValue.c_str(), "join"))
        {
            m_NextToken.assign(JOIN, "join");
        }
        else if (0 == strCaseCmp(theValue.c_str(), "order"))
        {
            m_NextToken.assign(ORDER, "order");
        }
        else if (0 == strCaseCmp(theValue.c_str(), "gap"))
        {
            m_NextToken.assign(GAP, "gap");
        }
        else
        {
            m_NextToken.assign(ACCESSION_NUMBER, theValue);
        }
    }

    void 
    LocationLexer::badToken(std::string const & theMessage)
    {
        if (m_ErrorLog)
        {
            m_ErrorLog->logError(m_CharIterator.currentLineNumber(),
                                 ERROR_BAD_TOKEN_IN_LOCATION,
                                 theMessage.c_str());
        }

        m_NextToken.assign(BAD_TOKEN, (char) currentChar());
    }

}
}
