/**++
 *   
 *   LICENSE
 *   -------
 *   
 *   Copyright (c) 2004 Renato Mancuso
 *   All rights reserved.
 *   
 *   Redistribution and use in source and binary forms, with or without modification, are 
 *   permitted provided that the following conditions are met:
 *   
 *   - Redistributions of source code must retain the above copyright notice, this list 
 *     of conditions and the following disclaimer.
 *   
 *   - Redistributions in binary form must reproduce the above copyright notice, this list
 *     of conditions and the following disclaimer in the documentation and/or other materials 
 *     provided with the distribution.
 *   
 *   - Neither the name of Renato Mancuso nor the names of its contributors may be used to 
 *     endorse or promote products derived from this software without specific prior written 
 *     permission.
 *   
 *   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS AS IS AND ANY EXPRESS 
 *   OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY 
 *   AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR 
 *   CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL 
 *   DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, 
 *   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER 
 *   IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT 
 *   OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *   
--**/


#include "OSLineParser.hpp"
#include "ParserErrors.hpp"
#include "ParserWarnings.hpp"
#include "StringUtils.hpp"

#include <algorithm>
#include <cstdlib>
#include <cassert>

namespace OpenEMBL
{
namespace Phoenix
{

    OSLineParser::OSLineParser(
        IParserCtx*             pParserCtx,
        IItemHandler<OSLine>*   pItemHandler)
        : ItemParserImpl<OSLine>(pParserCtx, pItemHandler)
    {
    }

    METHODIMP OSLineParser::parse()
    {
        long   theLineNumber = currentLineNumber();
        LPCSTR theLine       = currentLine();

        if (NULL == theLine)
        {
            logError(theLineNumber,
                     ERROR_EOF_FOUND,
                     NULL);
            return;
        }

        if (!isOSLine(theLine))
        {
            logError(theLineNumber,
                     ERROR_INVALID_OS_LINE,
                     theLine);
            nextLine();
            return;
        }

        StringList theLines;
        extractLines(isOSLine, theLines);
        processLines(theLines, theLineNumber);
    }

    void OSLineParser::processLines(const StringList & theLines, long theLineNumber)
    {
        OSLine theField;

        std::transform(
            theLines.begin(), theLines.end(),
            std::back_inserter(theField.Organism),
            getLineContent);

        notifyParsed(theLineNumber, theField);
    }

    bool OSLineParser::isOSLine(const char* theLine)
    {
        assert(NULL != theLine);
        return 0 == strncmp(theLine, "OS   ", 5);
    }

    String OSLineParser::getLineContent(const String & theLine)
    {
        assert(theLine.length() >= 5);
        return trim(theLine.substr(5));
    }


}
}
