/**++
 *   
 *   LICENSE
 *   -------
 *   
 *   Copyright (c) 2004 Renato Mancuso
 *   All rights reserved.
 *   
 *   Redistribution and use in source and binary forms, with or without modification, are 
 *   permitted provided that the following conditions are met:
 *   
 *   - Redistributions of source code must retain the above copyright notice, this list 
 *     of conditions and the following disclaimer.
 *   
 *   - Redistributions in binary form must reproduce the above copyright notice, this list
 *     of conditions and the following disclaimer in the documentation and/or other materials 
 *     provided with the distribution.
 *   
 *   - Neither the name of Renato Mancuso nor the names of its contributors may be used to 
 *     endorse or promote products derived from this software without specific prior written 
 *     permission.
 *   
 *   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS AS IS AND ANY EXPRESS 
 *   OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY 
 *   AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR 
 *   CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL 
 *   DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, 
 *   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER 
 *   IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT 
 *   OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *   
--**/


#include "ParserHost.hpp"
#include "ParserErrors.hpp"
#include "ParserWarnings.hpp"
#include "ParserErrorInfo.hpp"
#include "ParserWarningInfo.hpp"

#include <cstdio>
#include <sstream>
#include <stdexcept>
#include <typeinfo>
#include <cassert>

namespace OpenEMBL
{
namespace Phoenix
{

    ParserHost::ParserHost()
    {
        m_Parser.addListener(this);
    }

    ParserHost::~ParserHost() throw()
    {
    }

    METHODIMP ParserHost::parse()
    {
        ITextSource* theTextSource = NULL;

        try
        {
            m_EntryCount    = 0;
            theTextSource   = getTextSource();

            setParserOptions(m_ParserOptions);

            m_Parser.setOptions(m_ParserOptions);
            m_Parser.setTextSource(theTextSource);

            m_Parser.parse();

            if (0 == m_EntryCount)
            {
                onWarning(theTextSource != NULL? theTextSource->currentLineNumber() : 0,
                        WARNING_NO_ENTRY_FOUND,
                        NULL);
            }
        }
        catch (std::bad_alloc &)
        {
            onError(theTextSource != NULL? theTextSource->currentLineNumber() : 0,
                    ERROR_OUT_OF_MEMORY,
                    NULL);                    
        }
        catch (std::exception & e)
        {
            std::ostringstream oss;

            oss << "standard library exception ("
                << typeid(e).name()
                << "): "
                << e.what()
                ;
            
            onError(theTextSource != NULL? theTextSource->currentLineNumber() : 0,
                    ERROR_INTERNAL_ERROR_STDLIB_EXCEPTION,
                    oss.str().c_str());
        }
    }

    void ParserHost::setParserOptions(ParserOptions &)
    {
        /* NULL */;
    }

    void ParserHost::handleError(
        long          theLineNumber,
        int           theErrorCode,
        char const*   theDescription,
        char const*   theMessage)
    {
        assert(NULL != theDescription);

        if (NULL == theMessage)
            fprintf(stderr,
            "%s(%ld): error P%05d: %s.\n",
            getSourceDisplayName(),
            theLineNumber,
            theErrorCode,
            theDescription);
        else
            fprintf(stderr,
            "%s(%ld): error P%05d: %s: %s.\n",
            getSourceDisplayName(),
            theLineNumber,
            theErrorCode,
            theDescription,
            theMessage);
    }

    METHODIMP ParserHost::onError  (long theLineNumber, int theErrorCode, char const* theMessage)
    {
        ErrorInfo           theErrorInfo;
        ParserErrorInfo     theErrorMessage;

        if (!theErrorMessage.lookup(theErrorCode, theErrorInfo))
        {
            assert(false);
            handleError(theLineNumber, theErrorCode, "UNKNOWN ERROR", NULL);
        }
        else
        {
            handleError(theLineNumber, theErrorCode, theErrorInfo.Description, theMessage);
        }        
    }

    void ParserHost::handleWarning(
        long            theLineNumber,
        int             theWarningNumber,
        int             theWarningLevel,
        char const*    theDescription,
        char const*    theMessage)
    {
        assert(NULL != theDescription);

        if (NULL == theMessage)
            fprintf(stderr,
            "%s(%ld): warning W%05d: %s.\n",
            getSourceDisplayName(),
            theLineNumber,
            theWarningNumber,
            theDescription);
        else
            fprintf(stderr,
            "%s(%ld): warning W%05d: %s: %s.\n",
            getSourceDisplayName(),
            theLineNumber,
            theWarningNumber,
            theDescription,
            theMessage);
    }

    METHODIMP ParserHost::onWarning(long theLineNumber, int theWarningNumber, char const* theMessage)
    {
        WarningInfo         theWarningInfo;
        ParserWarningInfo   theWarningMessage;

        if (!theWarningMessage.lookup(theWarningNumber, theWarningInfo))
        {
            assert(false);
            handleWarning(theLineNumber, theWarningNumber, -1, "UNKNOWN WARNING", NULL);
        }
        else
        {
            handleWarning(theLineNumber, 
                theWarningNumber, 
                theWarningInfo.Severity,                
                theWarningInfo.Description, 
                theMessage);
        }        
    }

    METHODIMP ParserHost::onBeginEntry(long)
    {
        ++m_EntryCount;
    }

    METHODIMP ParserHost::onEndEntry()
    {
        /* NULL */;
    }

    METHODIMP ParserHost::onID(long, IDLine const &)
    {
        /* NULL */;
    }

    METHODIMP ParserHost::onAC(long, ACLine const &)
    {
        /* NULL */;
    }

    METHODIMP ParserHost::onSV(long, SVLine const &)
    {
        /* NULL */;
    }

    METHODIMP ParserHost::onDT(long, DTLine const &)
    {
        /* NULL */;
    }

    METHODIMP ParserHost::onDE(long, DELine const &)
    {
        /* NULL */;
    }

    METHODIMP ParserHost::onKW(long, KWLine const &)
    {
        /* NULL */;
    }

    METHODIMP ParserHost::onDR(long, DRLine const &)
    {
        /* NULL */;
    }

    METHODIMP ParserHost::onCC(long, CCLine const &)
    {
        /* NULL */;
    }

    METHODIMP ParserHost::onSQ(long, SQLine const &)
    {
        /* NULL */;
    }

    METHODIMP ParserHost::onXX(long, XXLine const &)
    {
        /* NULL */;
    }


    METHODIMP ParserHost::onBeginTaxonomyLines(long)
    {
        /* NULL */;
    }


    METHODIMP ParserHost::onOS(long, OSLine const &)    
    {
        /* NULL */;
    }


    METHODIMP ParserHost::onOC(long, OCLine const &)
    {
        /* NULL */;
    }


    METHODIMP ParserHost::onOG(long, OGLine const &)        
    {
        /* NULL */;
    }


    METHODIMP ParserHost::onEndTaxonomyLines()
    {
        /* NULL */;
    }


    METHODIMP ParserHost::onBeginFeatureTable(long)
    {
        /* NULL */;
    }

    METHODIMP ParserHost::onEndFeatureTable()
    {
        /* NULL */;
    }

    METHODIMP ParserHost::onBeginFeatureKey(long, FeatureKey const &)
    {
        /* NULL */;
    }


    METHODIMP ParserHost::onFeatureQualifier(long, FeatureQualifier const &)
    {
        /* NULL */;
    }


    METHODIMP ParserHost::onEndFeatureKey()
    {
        /* NULL */;
    }


    METHODIMP ParserHost::onBeginSequenceData(long)
    {
        /* NULL */;
    }


    METHODIMP ParserHost::onSequenceLine(long, SequenceLine const &)
    {
        /* NULL */;
    }


    METHODIMP ParserHost::onEndSequenceData()
    {
        /* NULL */;
    }


    METHODIMP ParserHost::onBeginTPALines(long)
    {
        /* NULL */;
    }


    METHODIMP ParserHost::onAS(long, ASLine const &)
    {
        /* NULL */;
    }


    METHODIMP ParserHost::onEndTPALines()
    {
        /* NULL */;
    }


    METHODIMP ParserHost::onBeginCOLines(long, bool)        
    {
        /* NULL */;
    }


    METHODIMP ParserHost::onCOSegmentInfo (long, COSegmentInfo const &)
    {
        /* NULL */;
    }


    METHODIMP ParserHost::onCOGapInfo (long, COGapInfo const &)
    {
        /* NULL */;
    }


    METHODIMP ParserHost::onEndCOLines()
    {
        /* NULL */;
    }


    METHODIMP ParserHost::onACStar(long, ACStarLine const &)
    {
        /* NULL */;
    }


    METHODIMP ParserHost::onBeginPublication(long)
    {
        /* NULL */;
    }


    METHODIMP ParserHost::onRN(long, RNLine const &)
    {
        /* NULL */;
    }


    METHODIMP ParserHost::onRC(long, RCLine const &)
    {
        /* NULL */;
    }


    METHODIMP ParserHost::onRP(long, RPLine const &)
    {
        /* NULL */;
    }


    METHODIMP ParserHost::onRX(long, RXLine const &)
    {
        /* NULL */;
    }


    METHODIMP ParserHost::onRG(long, RGLine const &)
    {
        /* NULL */;
    }


    METHODIMP ParserHost::onRA(long, RALine const &)
    {
        /* NULL */;
    }


    METHODIMP ParserHost::onRT(long, RTLine const &)
    {
        /* NULL */;
    }


    METHODIMP ParserHost::onSubmission        (long, SubmissionInfo         const &)
    {
        /* NULL */;
    }


    METHODIMP ParserHost::onUnpublished       (long, UnpublishedInfo        const &)
    {
        /* NULL */;
    }


    METHODIMP ParserHost::onBook              (long, BookInfo               const &)
    {
        /* NULL */;
    }


    METHODIMP ParserHost::onThesis            (long, ThesisInfo             const &)
    {
        /* NULL */;
    }


    METHODIMP ParserHost::onPatent            (long, PatentInfo             const &)
    {
        /* NULL */;    
    }


    METHODIMP ParserHost::onOnlineJournal     (long, OnlineJournalInfo      const &)
    {
        /* NULL */;
    }


    METHODIMP ParserHost::onElectronicResource(long, ElectronicResourceInfo const &)
    {
        /* NULL */;
    }

    
	METHODIMP ParserHost::onMiscResource(long, MiscResourceInfo const &)
    {
        /* NULL */;
    }


    METHODIMP ParserHost::onJournalArticle    (long, JournalArticleInfo     const &)
    {
        /* NULL */;
    }


    METHODIMP ParserHost::onEndPublication()
    {
        /* NULL */;
    }

}
}
