/**++
 *   
 *   LICENSE
 *   -------
 *   
 *   Copyright (c) 2004 Renato Mancuso
 *   All rights reserved.
 *   
 *   Redistribution and use in source and binary forms, with or without modification, are 
 *   permitted provided that the following conditions are met:
 *   
 *   - Redistributions of source code must retain the above copyright notice, this list 
 *     of conditions and the following disclaimer.
 *   
 *   - Redistributions in binary form must reproduce the above copyright notice, this list
 *     of conditions and the following disclaimer in the documentation and/or other materials 
 *     provided with the distribution.
 *   
 *   - Neither the name of Renato Mancuso nor the names of its contributors may be used to 
 *     endorse or promote products derived from this software without specific prior written 
 *     permission.
 *   
 *   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS AS IS AND ANY EXPRESS 
 *   OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY 
 *   AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR 
 *   CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL 
 *   DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, 
 *   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER 
 *   IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT 
 *   OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *   
--**/


#ifndef OPEN_EMBL_PARSER_HOST_INCLUDED
#define OPEN_EMBL_PARSER_HOST_INCLUDED

#if !defined(OPEN_EMBL_NO_PRAGMA_ONCE)
#pragma once
#endif

#include "NonCopyable.hpp"
#include "Parser.hpp"
#include "ParserOptions.hpp"
#include "ParserEvents.hpp"

namespace OpenEMBL
{
namespace Phoenix
{

    //
    // Base class to ease working with a Phoenix parser
    //

    class ParserHost 
        : public virtual IParserEventsListener
        , NonCopyable
    {
    public:
        ParserHost();
        virtual ~ParserHost() throw();

        virtual void parse();

        //
        // overridables
        //
    protected:        
        virtual void         setParserOptions(ParserOptions & theOptions);
        virtual ITextSource* getTextSource() = 0;
        virtual LPCSTR       getSourceDisplayName() = 0;

        virtual void handleError(
            long          theLineNumber,
            int           theErrorCode,
            char const*   theDescription,
            char const*   theMessage);

        virtual void handleWarning(
            long            theLineNumber,
            int             theWarningNumber,
            int             theWarningLevel,
            char const*     theDescription,
            char const*     theMessage);     

    private:
        ParserOptions   m_ParserOptions;     
        Parser          m_Parser;
        size_t          m_EntryCount;

        // ------------------------------------------------------------------
        //            ParserEvents interface canned implementation
        // ------------------------------------------------------------------
    public:
        // Errors/Warnings
        METHOD onError  (long theLineNumber, int theErrorCode, char const* theMessage);
        METHOD onWarning(long theLineNumber, int theWarningNumber, char const* theMessage);

        // Entry
        METHOD onBeginEntry(long theLineNumber);
        METHOD onEndEntry();

        // Individual Fields
        METHOD onID(long theLineNumber, IDLine const & theField);
        METHOD onAC(long theLineNumber, ACLine const & theField);
        METHOD onSV(long theLineNumber, SVLine const & theField);
        METHOD onDT(long theLineNumber, DTLine const & theField);
        METHOD onDE(long theLineNumber, DELine const & theField);
        METHOD onKW(long theLineNumber, KWLine const & theField);
        METHOD onDR(long theLineNumber, DRLine const & theField);
        METHOD onCC(long theLineNumber, CCLine const & theField);
        METHOD onSQ(long theLineNumber, SQLine const & theField);
        METHOD onXX(long theLineNumber, XXLine const & theField);

        // Taxonomy lines
        METHOD onBeginTaxonomyLines(long theLineNumber);
        METHOD onOS(long theLineNumber, OSLine const & theField);    
        METHOD onOC(long theLineNumber, OCLine const & theField);
        METHOD onOG(long theLineNumber, OGLine const & theField);        
        METHOD onEndTaxonomyLines();

        // Feature keys
        METHOD onBeginFeatureTable(long theLineNumber);
        METHOD onBeginFeatureKey(long theLineNumber, FeatureKey const & theField);
        METHOD onFeatureQualifier(long theLineNumber, FeatureQualifier const  & theQualifier);   
        METHOD onEndFeatureKey();
        METHOD onEndFeatureTable();

        // Sequence Data
        METHOD onBeginSequenceData(long theLineNumber);
        METHOD onSequenceLine(long theLineNumber, SequenceLine const & theField);
        METHOD onEndSequenceData();

        // TPA lines
        METHOD onBeginTPALines(long theLineNumber);
        METHOD onAS(long theLineNumber, ASLine const & theField);
        METHOD onEndTPALines();

        // CO lines
        METHOD onBeginCOLines(long theLineNumber, bool isComplement);        
        METHOD onCOSegmentInfo (long theLineNumber, COSegmentInfo const & theSegmentInfo);
        METHOD onCOGapInfo (long theLineNumber, COGapInfo const & theGapInfo);
        METHOD onEndCOLines();

        // Private line types
        METHOD onACStar(long theLineNumber, ACStarLine const & theField);

        // Publications
        METHOD onBeginPublication(long theLineNumber);

        METHOD onRN(long theLineNumber, RNLine const & theField);
        METHOD onRC(long theLineNumber, RCLine const & theField);
        METHOD onRP(long theLineNumber, RPLine const & theField);
        METHOD onRX(long theLineNumber, RXLine const & theField);
        METHOD onRG(long theLineNumber, RGLine const & theField);
        METHOD onRA(long theLineNumber, RALine const & theField);
        METHOD onRT(long theLineNumber, RTLine const & theField);

        //>> RL line types
        METHOD onSubmission        (long theLineNumber, SubmissionInfo         const & theField);
        METHOD onUnpublished       (long theLineNumber, UnpublishedInfo        const & theField);
        METHOD onBook              (long theLineNumber, BookInfo               const & theField);
        METHOD onThesis            (long theLineNumber, ThesisInfo             const & theField);
        METHOD onPatent            (long theLineNumber, PatentInfo             const & theField);
        METHOD onOnlineJournal     (long theLineNumber, OnlineJournalInfo      const & theField);
        METHOD onElectronicResource(long theLineNumber, ElectronicResourceInfo const & theField);
		METHOD onMiscResource      (long theLineNumber, MiscResourceInfo       const & theField);
        METHOD onJournalArticle    (long theLineNumber, JournalArticleInfo     const & theField);

        METHOD onEndPublication();
    };

}
} 

#endif // OPEN_EMBL_PARSER_HOST_INCLUDED
