/**++
 *   
 *   LICENSE
 *   -------
 *   
 *   Copyright (c) 2004 Renato Mancuso
 *   All rights reserved.
 *   
 *   Redistribution and use in source and binary forms, with or without modification, are 
 *   permitted provided that the following conditions are met:
 *   
 *   - Redistributions of source code must retain the above copyright notice, this list 
 *     of conditions and the following disclaimer.
 *   
 *   - Redistributions in binary form must reproduce the above copyright notice, this list
 *     of conditions and the following disclaimer in the documentation and/or other materials 
 *     provided with the distribution.
 *   
 *   - Neither the name of Renato Mancuso nor the names of its contributors may be used to 
 *     endorse or promote products derived from this software without specific prior written 
 *     permission.
 *   
 *   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS AS IS AND ANY EXPRESS 
 *   OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY 
 *   AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR 
 *   CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL 
 *   DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, 
 *   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER 
 *   IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT 
 *   OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *   
--**/


#include "ParserImpl.hpp"
#include "ParserWarnings.hpp"
#include "ParserErrors.hpp"
#include "EntryParser.hpp"
#include "AbortEntry.hpp"
#include "AbortRun.hpp"

#include <cstdlib>
#include <cassert>

namespace OpenEMBL
{
namespace Phoenix
{

    ParserImpl::ParserImpl()
        : m_pTextSource(NULL)
    {
    }

    ParserImpl::~ParserImpl() throw()
    {
    }

    void ParserImpl::setOptions(ParserOptions const & theOptions)
    {
        m_Options = theOptions;
    }

    void ParserImpl::setTextSource(ITextSource* theTextSource)
    {
        assert(NULL != theTextSource);
        m_pTextSource = theTextSource;
    }

    void ParserImpl::addListener(IParserEventsListener* theListener)
    {
        assert(NULL != theListener);

        if (NULL != theListener)
            m_Listeners.push_back(theListener);
    }

    METHODIMP_(LPCSTR) ParserImpl::currentLine() const NO_THROW
    {
        assert(NULL != m_pTextSource);

        return m_pTextSource->currentLine();
    }

    METHODIMP_(long) ParserImpl::currentLineNumber() const NO_THROW
    {
        assert(NULL != m_pTextSource);

        return m_pTextSource->currentLineNumber();
    }

    METHODIMP_(bool) ParserImpl::nextLine()
    {
        assert(NULL != m_pTextSource);

        return m_pTextSource->nextLine();
    }

    METHODIMP ParserImpl::logError(long theLineNumber, int theErrorCode, char const * theMessage)
    {
        broadcastError(theLineNumber, theErrorCode, theMessage);
    }

    METHODIMP ParserImpl::logWarning(long theLineNumber, int theWarning, char const * theMessage)
    {
        broadcastWarning(theLineNumber, theWarning, theMessage);
    }

    METHODIMP ParserImpl::parse()
    {
        if (NULL == m_pTextSource)
            return;

        initialize();

        //
        // if asked to do so, consume spurious error lines at beginning
        //
        if (m_Options.AllowErrorLinesAtBeginning)             
            goToNextEntry();

        try
        {
            while (NULL != currentLine())
            {
                try
                {
                    parseEntry();
                }
                catch(AbortEntry &)
                {                    
                    goToNextEntry();
                }
            }
        }
        catch(AbortRun &)
        {
            /* NULL */;
        }

        finalize();
    }

    void ParserImpl::initialize()
    {
    }

    void ParserImpl::finalize()
    {
    }

    void ParserImpl::goToNextEntry()
    {
        for (;;)
        {
            if (NULL == currentLine())
                break;

            if (0 == strncmp(currentLine(), "ID   ", 5))
                break;

            nextLine();
        }
    }

    void ParserImpl::parseEntry()
    {
        EntryParser theParser(this, this);
        theParser.parse();
    }

} 
}
