/**++
 *   
 *   LICENSE
 *   -------
 *   
 *   Copyright (c) 2004 Renato Mancuso
 *   All rights reserved.
 *   
 *   Redistribution and use in source and binary forms, with or without modification, are 
 *   permitted provided that the following conditions are met:
 *   
 *   - Redistributions of source code must retain the above copyright notice, this list 
 *     of conditions and the following disclaimer.
 *   
 *   - Redistributions in binary form must reproduce the above copyright notice, this list
 *     of conditions and the following disclaimer in the documentation and/or other materials 
 *     provided with the distribution.
 *   
 *   - Neither the name of Renato Mancuso nor the names of its contributors may be used to 
 *     endorse or promote products derived from this software without specific prior written 
 *     permission.
 *   
 *   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS AS IS AND ANY EXPRESS 
 *   OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY 
 *   AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR 
 *   CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL 
 *   DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, 
 *   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER 
 *   IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT 
 *   OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *   
--**/


#include "PatentInfoParser.hpp"
#include "ParserErrors.hpp"
#include "ParserWarnings.hpp"
#include "StringUtils.hpp"
#include "ListReader.hpp"
#include "RLLineCodes.hpp"
#include "Regex.hpp"

#include <algorithm>
#include <cstdlib>
#include <sstream>
#include <cassert>

namespace OpenEMBL
{
namespace Phoenix
{

    PatentInfoParser::PatentInfoParser(
        IParserCtx*                 pParserCtx,
        IItemHandler<PatentInfo>*   pItemHandler)
        : ItemParserImpl<PatentInfo>(pParserCtx, pItemHandler)
    {
    }

    METHODIMP PatentInfoParser::parse()
    {
        long   theLineNumber = currentLineNumber();
        LPCSTR theLine       = currentLine();

        if (NULL == theLine)
        {
            logError(theLineNumber,
                     ERROR_EOF_FOUND,
                     NULL);
            return;
        }

        if (0 != strNCaseCmp(theLine, RL_PATENT, strlen(RL_PATENT)))
        {
            logError(theLineNumber,
                     ERROR_INVALID_RL_PATENT_LINE,
                     theLine);
            nextLine();
            return;
        }

        StringList theLines;
        extractLines(isRLLine, theLines);
        processLines(theLines, theLineNumber);
    }

    bool PatentInfoParser::isRLLine(const char* theLine)
    {
        assert(NULL != theLine);
        return 0 == strncmp(theLine, "RL   ", 5);
    }

    String PatentInfoParser::getLineContent(const String & theLine)
    {
        assert(theLine.length() >= 5);
        return trimRight(theLine.substr(5));
    }

    void PatentInfoParser::processLines(const StringList & theLines, long theLineNumber)
    {
        if (theLines.size() == 0)
        {
            logError(theLineNumber,
                     ERROR_MISSING_PATENT_INFO,
                     NULL);
            return;
        }

        /*

        apparently patent applicant info lines are optional...oh, well

        if (theLines.size() == 1)
        {
            logError(theLineNumber + 1,
                     ERROR_MISSING_PATENT_APPLICANT_INFO,
                     NULL);
            return;
        }
        */
        
        PatentInfo theField;

        if (!processHeader(theLines[0], theLineNumber, theField))
            return;

        if (!processApplicants(theLines, theLineNumber + 1, theField))
            return;

        notifyParsed(theLineNumber, theField);
    }

    bool PatentInfoParser::processHeader(
        String const &      theLine, 
        long                theLineNumber, 
        PatentInfo &        theField)
    {
        //
        // Regex is: ^RL\s{3}Patent number\s\s*(.+),\s*([^.]+)(\.){0,1}\s*$
        //
        // groups:
        //  1) Patent number
        //  2) date
        //  3) final dot
        //
        static Regex theRegex("^RL\\s{3}Patent number\\s\\s*(.+),\\s*([^.]+)(\\.){0,1}\\s*$",
                              PCRE_CASELESS);

        Match theMatches;

        if (!regexSearch(theLine, theMatches, theRegex))
        {
            logError(currentLineNumber(),
                     ERROR_INVALID_RL_PATENT_LINE,
                     theLine.c_str());
            return false;
        }

        theField.PatentNumber = trim(theMatches.str(1));
        theField.Date         = trim(theMatches.str(2));

        if (!theMatches[3].matched)
        {
            logWarning(theLineNumber,
                       WARNING_MISSING_FINAL_DOT,
                       theLine.c_str());
        }

        return true;
    }

    String PatentInfoParser::makeString(StringList const & theLines)
    {
        std::ostringstream oss;

        if (theLines.size() > 0)
        {
            oss << trimLeft(theLines[0]);

            for (size_t i = 1; i < theLines.size(); ++i)
                oss << ' ' << trimLeft(theLines[i]);
        }
        
        return oss.str();
    }

    bool PatentInfoParser::processApplicants(
                        StringList const &  theLines, 
                        long                theLineNumber, 
                        PatentInfo &        theField)
    {
        if (theLines.size() == 1)
            return true;

        StringList theApplicantLines;

        std::transform( theLines.begin() + 1,
                        theLines.end(),
                        std::back_inserter(theApplicantLines),
                        getLineContent );
        //
        // oddly, a patent applicant can span more than 1 line.
        // so, we need to first concatenate all strings together
        // and then get the list...
        //

        String theList = makeString(theApplicantLines);

        ListReader theListBuilder(this, ';', '.');
        theListBuilder.readList(theLineNumber, theList, theField.Applicants);
        return true;
    }

}
}
