/**++
 *   
 *   LICENSE
 *   -------
 *   
 *   Copyright (c) 2004 Renato Mancuso
 *   All rights reserved.
 *   
 *   Redistribution and use in source and binary forms, with or without modification, are 
 *   permitted provided that the following conditions are met:
 *   
 *   - Redistributions of source code must retain the above copyright notice, this list 
 *     of conditions and the following disclaimer.
 *   
 *   - Redistributions in binary form must reproduce the above copyright notice, this list
 *     of conditions and the following disclaimer in the documentation and/or other materials 
 *     provided with the distribution.
 *   
 *   - Neither the name of Renato Mancuso nor the names of its contributors may be used to 
 *     endorse or promote products derived from this software without specific prior written 
 *     permission.
 *   
 *   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS AS IS AND ANY EXPRESS 
 *   OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY 
 *   AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR 
 *   CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL 
 *   DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, 
 *   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER 
 *   IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT 
 *   OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *   
--**/


#include "PublicationLineParser.hpp"
#include "ParserErrors.hpp"
#include "ParserWarnings.hpp"
#include "RNLineParser.hpp"
#include "RCLineParser.hpp"
#include "RPLineParser.hpp"
#include "RXLineParser.hpp"
#include "RALineParser.hpp"
#include "RGLineParser.hpp"
#include "RTLineParser.hpp"
#include "RLLineParser.hpp"

namespace OpenEMBL
{
namespace Phoenix
{

    PublicationLineParser::PublicationLineParser(
        IParserCtx*             pParserCtx,
        IPublicationHandler*    pPublicationHandler )
        : m_pParserCtx(pParserCtx)
        , m_pPublicationHandler(pPublicationHandler)
    {
    }

    METHODIMP_(LPCSTR) PublicationLineParser::currentLine() const NO_THROW
    {
        if (NULL == m_pParserCtx)
            return NULL;

        return m_pParserCtx->currentLine();
    }

    METHODIMP_(long) PublicationLineParser::currentLineNumber() const NO_THROW
    {
        if (NULL == m_pParserCtx)
            return TEXTSOURCE_INVALID_LINE_NUMBER;

        return m_pParserCtx->currentLineNumber();
    }

    METHODIMP_(bool) PublicationLineParser::nextLine()
    {
        if (NULL == m_pParserCtx)
            return false;

        return m_pParserCtx->nextLine();
    }

    METHODIMP PublicationLineParser::logError(long theLineNumber, int theErrorCode, char const * theMessage)
    {
        if (NULL != m_pParserCtx)
            m_pParserCtx->logError(theLineNumber, theErrorCode, theMessage);
    }

    METHODIMP PublicationLineParser::logWarning(long theLineNumber, int theWarning, char const * theMessage)
    {
        if (NULL != m_pParserCtx)
            m_pParserCtx->logWarning(theLineNumber, theWarning, theMessage);
    }

    bool PublicationLineParser::peekLine(const char* theHeader)
    {
        return NULL != currentLine() 
               &&
               0 == strncmp(currentLine(), theHeader, strlen(theHeader));
    }

    void PublicationLineParser::notifyBeginPublication(long theLineNumber)
    {
        if (NULL != m_pPublicationHandler)
            m_pPublicationHandler->onBeginPublication(theLineNumber);
    }

    void PublicationLineParser::notifyEndPublication()
    {
        if (NULL != m_pPublicationHandler)
            m_pPublicationHandler->onEndPublication();
    }

    METHODIMP PublicationLineParser::parse()
    {
        bool hasRGLine = false;

        notifyBeginPublication(currentLineNumber());

        // RN
        if (NULL != currentLine())
        {
            RNLineParser theParser(this, m_pPublicationHandler);
            theParser.parse();
        }

        if (peekLine("RC   "))
        {
            RCLineParser theParser(this, m_pPublicationHandler);                
            theParser.parse();
        }

        if (peekLine("RP   "))
        {
            RPLineParser theParser(this, m_pPublicationHandler);
            theParser.parse();
        }

        while (peekLine("RX   "))
        {
            RXLineParser theParser(this, m_pPublicationHandler);
            theParser.parse();
        }

        if (peekLine("RG   "))
        {
            RGLineParser theParser(this, m_pPublicationHandler);
            theParser.parse();
            hasRGLine = true;
        }

        if (peekLine("RA   "))
        {
            RALineParser theParser(this, m_pPublicationHandler);
            theParser.parse();
        }
        else if (!hasRGLine)
        {
            logError(currentLineNumber(),
                     ERROR_MISSING_RA_LINE,
                     currentLine());
        }

        if (peekLine("RT   "))
        {
            RTLineParser theParser(this, m_pPublicationHandler);
            theParser.parse();
        }
        else
        {
            logError(currentLineNumber(),
                     ERROR_MISSING_RT_LINE,
                     currentLine());
        }

        if (peekLine("RL   "))
        {
            RLLineParser theParser(this, m_pPublicationHandler);
            theParser.parse();
        }
        else
        {
            logError(currentLineNumber(),
                     ERROR_MISSING_RL_LINE,
                     currentLine());
        }

        notifyEndPublication();
    }   

}
}
