/**++
 *   
 *   LICENSE
 *   -------
 *   
 *   Copyright (c) 2004 Renato Mancuso
 *   All rights reserved.
 *   
 *   Redistribution and use in source and binary forms, with or without modification, are 
 *   permitted provided that the following conditions are met:
 *   
 *   - Redistributions of source code must retain the above copyright notice, this list 
 *     of conditions and the following disclaimer.
 *   
 *   - Redistributions in binary form must reproduce the above copyright notice, this list
 *     of conditions and the following disclaimer in the documentation and/or other materials 
 *     provided with the distribution.
 *   
 *   - Neither the name of Renato Mancuso nor the names of its contributors may be used to 
 *     endorse or promote products derived from this software without specific prior written 
 *     permission.
 *   
 *   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS AS IS AND ANY EXPRESS 
 *   OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY 
 *   AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR 
 *   CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL 
 *   DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, 
 *   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER 
 *   IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT 
 *   OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *   
--**/


#include "RLLineParser.hpp"
#include "ParserErrors.hpp"
#include "SubmissionInfoParser.hpp"
#include "UnpublishedInfoParser.hpp"
#include "ThesisInfoParser.hpp"
#include "OnlineJournalInfoParser.hpp"
#include "ElectronicResourceInfoParser.hpp"
#include "MiscResourceInfoParser.hpp"
#include "BookInfoParser.hpp"
#include "PatentInfoParser.hpp"
#include "JournalArticleInfoParser.hpp"
#include "StringUtils.hpp"
#include "RLLineCodes.hpp"

#include <cstring>
#include <cstdlib>
#include <cassert>

namespace OpenEMBL
{
namespace Phoenix
{
    
    /* class static */
    RLLineParser::HANDLER_REC RLLineParser::ms_handlers[] = {
        {RL_SUBMITTED,              &RLLineParser::handleSubmitted},
        {RL_UNPUBLISHED,            &RLLineParser::handleUnpublished},
        {RL_PATENT,                 &RLLineParser::handlePatent},
        {RL_ONLINE_PUBLICATION,     &RLLineParser::handleOnlinePublication},
        {RL_THESIS,                 &RLLineParser::handleThesis},
        {RL_BOOK,                   &RLLineParser::handleBook},
        {RL_ELECTRONIC_RESOURCE,    &RLLineParser::handleElectronicResource},
		{RL_MISC_RESOURCE,			&RLLineParser::handleMiscResource},
        {RL_JOURNAL,                &RLLineParser::handleJournal},
        {NULL,                      0}
    };

    bool RLLineParser::lookupHandler(
        char const*    theLine, 
        HANDLER_REC*   theHandlerRec)
    {
        if (NULL == theLine)
            return false;

        for (size_t i = 0; ; ++i)
        {
            HANDLER_REC & theItem = ms_handlers[i];

            if (NULL == theItem.Handler)
                break;

            if (0 == strNCaseCmp(theLine, theItem.Header, strlen(theItem.Header)))
            {
                if (NULL != theHandlerRec)
                    *theHandlerRec = theItem;

                return true;
            }
        }

        return false;
    }

    RLLineParser::RLLineParser(
        IParserCtx*             pParserCtx,
        IPublicationHandler*    pPublicationHandler)
        : m_pParserCtx(pParserCtx)
        , m_pPublicationHandler(pPublicationHandler)
    {
    }

    METHODIMP_(LPCSTR) RLLineParser::currentLine() const NO_THROW
    {
        return m_pParserCtx != NULL ? m_pParserCtx->currentLine() : NULL;
    }

    METHODIMP_(long) RLLineParser::currentLineNumber() const NO_THROW
    {
        return m_pParserCtx != NULL ? m_pParserCtx->currentLineNumber() : TEXTSOURCE_INVALID_LINE_NUMBER;
    }

    METHODIMP_(bool) RLLineParser::nextLine()
    {
        return m_pParserCtx != NULL ? m_pParserCtx->nextLine() : false;
    }

    METHODIMP RLLineParser::logError(long theLineNumber, int theErrorCode, LPCSTR theMessage)
    {
        if (NULL != m_pParserCtx)
            m_pParserCtx->logError(theLineNumber, theErrorCode, theMessage);
    }

    METHODIMP RLLineParser::logWarning(long theLineNumber, int theWarning, LPCSTR theMessage)
    {
        if (NULL != m_pParserCtx)
            m_pParserCtx->logWarning(theLineNumber, theWarning, theMessage);
    }

    bool RLLineParser::isRLLine(const char* theLine)
    {
        assert(NULL != theLine);
        return 0 == strncmp(theLine, "RL   ", 5);
    }

    METHODIMP RLLineParser::parse()
    {
        long   theLineNumber = currentLineNumber();
        LPCSTR theLine       = currentLine();

        if (NULL == theLine)
        {
            logError(theLineNumber,
                ERROR_EOF_FOUND,
                NULL);
            return;
        }

        if (!isRLLine(theLine))
        {
            logError(theLineNumber,
                     ERROR_INVALID_RL_LINE,
                     theLine);
            nextLine();
            return;
        }
        
        HANDLER_REC theHandlerRec;

        if (!lookupHandler(theLine, &theHandlerRec))
        {
            assert(0); // have you forgotten to register the handlers?
            logError(theLineNumber,
                    ERROR_UNKNOWN_LINE_TYPE,
                    theLine);
            nextLine();
            return;
        }

        (this->*theHandlerRec.Handler)();
    }

    void RLLineParser::nullHandler()
    {
        while (isRLLine(currentLine()))
            nextLine();
    }

    void RLLineParser::handleSubmitted()
    {
        assert(NULL != currentLine());
        assert(0 == strncmp(currentLine(), RL_SUBMITTED, strlen(RL_SUBMITTED)));

        SubmissionInfoParser theParser(this, m_pPublicationHandler);
        theParser.parse();
    }

    void RLLineParser::handleUnpublished()
    {
        assert(NULL != currentLine());
        assert(0 == strncmp(currentLine(), RL_UNPUBLISHED, strlen(RL_UNPUBLISHED)));

        UnpublishedInfoParser theParser(this, m_pPublicationHandler);
        theParser.parse();
    }

    void RLLineParser::handleBook()
    {
        assert(NULL != currentLine());
        assert(0 == strncmp(currentLine(), RL_BOOK, strlen(RL_BOOK)));

        BookInfoParser theParser(this, m_pPublicationHandler);
        theParser.parse();
    }

    void RLLineParser::handleThesis()
    {
        assert(NULL != currentLine());
        assert(0 == strncmp(currentLine(), RL_THESIS, strlen(RL_THESIS)));

        ThesisInfoParser theParser(this, m_pPublicationHandler);
        theParser.parse();
    }

    void RLLineParser::handlePatent()
    {
        assert(NULL != currentLine());
        assert(0 == strncmp(currentLine(), RL_PATENT, strlen(RL_PATENT)));

        PatentInfoParser theParser(this, m_pPublicationHandler);
        theParser.parse();
    }

    void RLLineParser::handleOnlinePublication()
    {
        assert(NULL != currentLine());
        assert(0 == strncmp(currentLine(), RL_ONLINE_PUBLICATION, strlen(RL_ONLINE_PUBLICATION)));

        OnlineJournalInfoParser theParser(this, m_pPublicationHandler);
        theParser.parse();
    }

    void RLLineParser::handleElectronicResource()
    {
        assert(NULL != currentLine());
        assert(0 == strncmp(currentLine(), RL_ELECTRONIC_RESOURCE, strlen(RL_ELECTRONIC_RESOURCE)));

        ElectronicResourceInfoParser theParser(this, m_pPublicationHandler);
        theParser.parse();
    }

	void RLLineParser::handleMiscResource()
	{
        assert(NULL != currentLine());
        assert(0 == strncmp(currentLine(), RL_MISC_RESOURCE, strlen(RL_MISC_RESOURCE)));

        MiscResourceInfoParser theParser(this, m_pPublicationHandler);
        theParser.parse();		
	}

    void RLLineParser::handleJournal()
    {
        assert(NULL != currentLine());
        assert(0 == strncmp(currentLine(), RL_JOURNAL, strlen(RL_JOURNAL)));

        JournalArticleInfoParser theParser(this, m_pPublicationHandler);
        theParser.parse();
    }

}
}

