/**++
 *   
 *   LICENSE
 *   -------
 *   
 *   Copyright (c) 2004 Renato Mancuso
 *   All rights reserved.
 *   
 *   Redistribution and use in source and binary forms, with or without modification, are 
 *   permitted provided that the following conditions are met:
 *   
 *   - Redistributions of source code must retain the above copyright notice, this list 
 *     of conditions and the following disclaimer.
 *   
 *   - Redistributions in binary form must reproduce the above copyright notice, this list
 *     of conditions and the following disclaimer in the documentation and/or other materials 
 *     provided with the distribution.
 *   
 *   - Neither the name of Renato Mancuso nor the names of its contributors may be used to 
 *     endorse or promote products derived from this software without specific prior written 
 *     permission.
 *   
 *   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS AS IS AND ANY EXPRESS 
 *   OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY 
 *   AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR 
 *   CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL 
 *   DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, 
 *   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER 
 *   IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT 
 *   OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *   
--**/


#include "RPLineParser.hpp"
#include "ParserErrors.hpp"
#include "ParserWarnings.hpp"
#include "StringUtils.hpp"
#include "ListReader.hpp"
#include "Regex.hpp"

#include <algorithm>
#include <cstdlib>
#include <cassert>

namespace OpenEMBL
{
namespace Phoenix
{

    RPLineParser::RPLineParser(
        IParserCtx*             pParserCtx,
        IItemHandler<RPLine>*   pItemHandler)
        : ItemParserImpl<RPLine>(pParserCtx, pItemHandler)
    {
    }

    METHODIMP RPLineParser::parse()
    {
        long   theLineNumber = currentLineNumber();
        LPCSTR theLine       = currentLine();

        if (NULL == theLine)
        {
            logError(theLineNumber,
                     ERROR_EOF_FOUND,
                     NULL);
            return;
        }

        if (!isRPLine(theLine))
        {
            logError(theLineNumber,
                     ERROR_INVALID_RP_LINE,
                     theLine);
            nextLine();
            return;
        }

        StringList theLines;
        extractLines(isRPLine, theLines);
        processLines(theLines, theLineNumber);
    }

    bool RPLineParser::isRPLine(const char* theLine)
    {
        assert(NULL != theLine);
        return 0 == strncmp(theLine, "RP   ", 5);
    }

    String RPLineParser::getLineContent(const String & theLine)
    {
        assert(theLine.length() >= 5);
        return trimRight(theLine.substr(5));
    }

    void RPLineParser::processLines(const StringList & theLines, long theLineNumber)
    {
        RPLine theField;

        for (size_t i = 0; i < theLines.size(); ++i)
        {
            if (!processLine(getLineContent(theLines[i]), theLineNumber + long(i), theField))
                return;
        }

        notifyParsed(theLineNumber, theField);
    }

    bool RPLineParser::processLine(const std::string & theLine, long theLineNumber, RPLine & theField)
    {
        ListReader theListBuilder(this, ',', 0, ListReader::LACK_TERMINATOR);
        StringList theRanges;

        theListBuilder.readList(theLineNumber, theLine, theRanges);

        for (size_t i = 0; i < theRanges.size(); ++i)
        {
            Range theRange;

            if (!processRange(theRanges[i], theRange))
            {
                logError(theLineNumber,
                         ERROR_INVALID_RP_LINE_RANGE,
                         theRanges[i].c_str()
                         );

                return false;
            }            

            theField.Positions.push_back(theRange);
        } 

        return true;
    }

    bool RPLineParser::processRange(const std::string & theToken, Range & theRange)
    {
        //
        //  Regex is: ^(\d+)\s*-\s*(\d+)$
        //
        static Regex theRegex("^(\\d+)\\s*-\\s*(\\d+)$");

        Match theMatches;

        if (!regexSearch(theToken, theMatches, theRegex))
            return false;

        theRange.First = atoi(theMatches.str(1).c_str());
        theRange.Last  = atoi(theMatches.str(2).c_str());

        return true;
    }

}
}
