/**++
 *   
 *   LICENSE
 *   -------
 *   
 *   Copyright (c) 2004 Renato Mancuso
 *   All rights reserved.
 *   
 *   Redistribution and use in source and binary forms, with or without modification, are 
 *   permitted provided that the following conditions are met:
 *   
 *   - Redistributions of source code must retain the above copyright notice, this list 
 *     of conditions and the following disclaimer.
 *   
 *   - Redistributions in binary form must reproduce the above copyright notice, this list
 *     of conditions and the following disclaimer in the documentation and/or other materials 
 *     provided with the distribution.
 *   
 *   - Neither the name of Renato Mancuso nor the names of its contributors may be used to 
 *     endorse or promote products derived from this software without specific prior written 
 *     permission.
 *   
 *   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS AS IS AND ANY EXPRESS 
 *   OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY 
 *   AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR 
 *   CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL 
 *   DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, 
 *   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER 
 *   IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT 
 *   OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *   
--**/


#include "RTLineParser.hpp"
#include "ParserErrors.hpp"
#include "ParserWarnings.hpp"
#include "StringUtils.hpp"

#include <algorithm>
#include <cstdlib>
#include <cassert>


namespace OpenEMBL
{
namespace Phoenix
{

    RTLineParser::RTLineParser(
        IParserCtx*             pParserCtx,
        IItemHandler<RTLine>*   pItemHandler)
        : ItemParserImpl<RTLine>(pParserCtx, pItemHandler)
    {
    }

    METHODIMP RTLineParser::parse()
    {
        long   theLineNumber = currentLineNumber();
        LPCSTR theLine       = currentLine();

        if (NULL == theLine)
        {
            logError(theLineNumber,
                     ERROR_EOF_FOUND,
                     NULL);
            return;
        }

        if (!isRTLine(theLine))
        {
            logError(theLineNumber,
                     ERROR_INVALID_RT_LINE,
                     theLine);
            nextLine();
            return;
        }

        StringList theLines;
        extractLines(isRTLine, theLines);        
        processLines(theLines, theLineNumber);
    }

    void RTLineParser::processLines(const StringList & theLines, long theLineNumber)
    {
        assert(theLines.size() > 0);

        RTLine theField;       

        std::transform(
            theLines.begin(), theLines.end(),
            std::back_inserter(theField.Text),
            getLineContent);

        removeDelimiters(theField.Text, theLineNumber);

        notifyParsed(theLineNumber, theField);
    }

    bool RTLineParser::isRTLine(const char* theLine)
    {
        assert(NULL != theLine);
        return 0 == strncmp(theLine, "RT   ", 5);
    }

    String RTLineParser::getLineContent(const String & theLine)
    {
        assert(theLine.length() >= 5);
        return trimRight(theLine.substr(5));
    }

    void RTLineParser::removeDelimiters(StringList & theLines, long theLineNumber)
    {
        //
        // remove terminal ';' and terminal '"' character from last line 
        //
        String & theLastLine = theLines[theLines.size() - 1];

        if (theLastLine.empty() || ';' != theLastLine[theLastLine.length() - 1])
        {
            logWarning(
                theLineNumber + (long) (theLines.size() - 1),
                WARNING_MISSING_RT_LINE_TERMINATOR,
                theLastLine.c_str());
        }
        else
        {
            theLastLine.erase(theLastLine.length() - 1);
            theLastLine = trimRight(theLastLine);
        }

        if (!theLastLine.empty())
        {
            if ('"' != theLastLine[theLastLine.length() - 1])
            {
                
                logWarning(
                    theLineNumber + (long) (theLines.size() - 1),
                    WARNING_MISSING_RT_LINE_FINAL_QUOTE,
                    theLastLine.c_str());
            }
            else
            {
                theLastLine.erase(theLastLine.length() - 1);
            }
        }

        //
        // remove initial '"' from first line
        //
        String & theFirstLine = theLines[0];

        if (!theFirstLine.empty())
        {
            if ('"' != theFirstLine[0])
            {
                logWarning(theLineNumber,
                        WARNING_MISSING_RT_LINE_INITIAL_QUOTE,
                        theFirstLine.c_str());                      
            }
            else
            {
                theFirstLine.erase(0, 1);
            }
        }
    }

}
}
