/**++
 *   
 *   LICENSE
 *   -------
 *   
 *   Copyright (c) 2004 Renato Mancuso
 *   All rights reserved.
 *   
 *   Redistribution and use in source and binary forms, with or without modification, are 
 *   permitted provided that the following conditions are met:
 *   
 *   - Redistributions of source code must retain the above copyright notice, this list 
 *     of conditions and the following disclaimer.
 *   
 *   - Redistributions in binary form must reproduce the above copyright notice, this list
 *     of conditions and the following disclaimer in the documentation and/or other materials 
 *     provided with the distribution.
 *   
 *   - Neither the name of Renato Mancuso nor the names of its contributors may be used to 
 *     endorse or promote products derived from this software without specific prior written 
 *     permission.
 *   
 *   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS AS IS AND ANY EXPRESS 
 *   OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY 
 *   AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR 
 *   CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL 
 *   DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, 
 *   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER 
 *   IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT 
 *   OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *   
--**/


#include "RXLineParser.hpp"
#include "ParserErrors.hpp"
#include "ParserWarnings.hpp"
#include "StringUtils.hpp"
#include "StringTokenizer.hpp"

#include <vector>
#include <string>
#include <cstdlib>
#include <cassert>

namespace OpenEMBL
{
namespace Phoenix
{

    RXLineParser::RXLineParser(
        IParserCtx*             pParserCtx,
        IItemHandler<RXLine>*   pItemHandler)
        : ItemParserImpl<RXLine>(pParserCtx, pItemHandler)
    {
    }

    METHODIMP RXLineParser::parse()
    {
        long   theLineNumber = currentLineNumber();
        LPCSTR theLine       = currentLine();

        if (NULL == theLine)
        {
            logError(theLineNumber,
                     ERROR_EOF_FOUND,
                     NULL );
            return;
        }

        processLine(theLine, theLineNumber);

        nextLine();
    }

    void RXLineParser::processLine(const std::string & theLine, long theLineNumber)
    {
        if (0 != strncmp(theLine.c_str(), "RX   ", 5))    
        {
            logError(theLineNumber,
                     ERROR_INVALID_RX_LINE,
                     theLine.c_str());
            return;
        }

        StringTokenizer             theTokenizer( " " );
        std::vector<std::string>    theIdentifiers;

        theTokenizer.tokenize( theLine.substr(5), theIdentifiers );

        if (theIdentifiers.size() != 2)
        {
            logError( theLineNumber,
                      ERROR_INVALID_RX_LINE,
                      theLine.c_str() );
            return;
        }

        removeTerminator( theIdentifiers[0], ';' );
        removeTerminator( theIdentifiers[1], '.' );

        RXLine theField;

        theField.DatabaseIdentifier = theIdentifiers[0];
        theField.Identifier         = theIdentifiers[1];

        notifyParsed(theLineNumber, theField);
    }

}
}
