/**++
 *   
 *   LICENSE
 *   -------
 *   
 *   Copyright (c) 2004 Renato Mancuso
 *   All rights reserved.
 *   
 *   Redistribution and use in source and binary forms, with or without modification, are 
 *   permitted provided that the following conditions are met:
 *   
 *   - Redistributions of source code must retain the above copyright notice, this list 
 *     of conditions and the following disclaimer.
 *   
 *   - Redistributions in binary form must reproduce the above copyright notice, this list
 *     of conditions and the following disclaimer in the documentation and/or other materials 
 *     provided with the distribution.
 *   
 *   - Neither the name of Renato Mancuso nor the names of its contributors may be used to 
 *     endorse or promote products derived from this software without specific prior written 
 *     permission.
 *   
 *   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS AS IS AND ANY EXPRESS 
 *   OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY 
 *   AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR 
 *   CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL 
 *   DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, 
 *   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER 
 *   IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT 
 *   OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *   
--**/

#include "Regex.hpp"

#include <sstream>
#include <cassert>

namespace OpenEMBL
{
namespace Phoenix
{


    Regex::Regex( const char* re, unsigned int options )
        : _re(0)
        , _study(0)
    {
        assert( re != 0 );
        compile( re, options );
    }


    Regex::~Regex()
    {
        if ( _re )      
            pcre_free( _re );

        if ( _study )   
            pcre_free( _study );
    }


    void 
    Regex::compile( const char* re, unsigned options )
    {
	    const char * errorPtr;
	    int          errorOffset;
    	
	    _re = pcre_compile( re, (int) options, &errorPtr, &errorOffset, 0 );

	    if ( !_re )
        {
            std::ostringstream oss;
                
            oss << "Invalid pattern '" 
                << re
                << "': error at position "
                << errorOffset
                << ": "
                << errorPtr;

		    throw exception( oss.str() );
        }

        const char* errMsg = 0;

        _study = pcre_study( _re, 0, &errMsg );

        if ( errMsg != 0 )
        {
            std::ostringstream oss;

            oss << "Failed to study pattern '"
                << re
                << "': "
                << errMsg;

            throw exception( oss.str() );
        }
    }


    void 
    Regex::check( int errorCode )
    {
        if ( errorCode >= 0 )
            return;

        switch (errorCode)
        {
            case PCRE_ERROR_NOMATCH        : throw exception( "PCRE_ERROR_NOMATCH" );
            case PCRE_ERROR_NULL           : throw exception( "PCRE_ERROR_NULL" );
            case PCRE_ERROR_BADOPTION      : throw exception( "PCRE_ERROR_BADOPTION" );
            case PCRE_ERROR_BADMAGIC       : throw exception( "PCRE_ERROR_BADMAGIC" );
            case PCRE_ERROR_UNKNOWN_NODE   : throw exception( "PCRE_ERROR_UNKNOWN_NODE" );
            case PCRE_ERROR_NOMEMORY       : throw exception( "PCRE_ERROR_NOMEMORY" );
            case PCRE_ERROR_NOSUBSTRING    : throw exception( "PCRE_ERROR_NOSUBSTRING" );
            case PCRE_ERROR_BADUTF8        : throw exception( "PCRE_ERROR_BADUTF8" );
            case PCRE_ERROR_BADUTF8_OFFSET : throw exception( "PCRE_ERROR_BADUTF8_OFFSET" );
            case PCRE_ERROR_PARTIAL        : throw exception( "PCRE_ERROR_PARTIAL" );
            case PCRE_ERROR_BADPARTIAL     : throw exception( "PCRE_ERROR_BADPARTIAL" );
            case PCRE_ERROR_BADCOUNT       : throw exception( "PCRE_ERROR_BADCOUNT" );
            case PCRE_ERROR_INTERNAL       : throw exception( "PCRE_ERROR_INTERNAL" );

            default:
                {
                    std::ostringstream oss;
                    oss << "Unknown PCRE error code " << errorCode;
                    throw exception( oss.str() );
                }
        }
    }


    int  
    Regex::getCaptureCount() const
    {
        int value = 0;
        check( pcre_fullinfo( _re, _study, PCRE_INFO_CAPTURECOUNT, &value ) );
        return value;
    }


    Match 
    Regex::find( std::string const & line ) const
    {
        int vsize = 3 * ( getCaptureCount() + 1 );

        std::vector<int> ovector( (size_t) vsize, (int) Match::INVALID_INDEX_VALUE );

        int errorCode = pcre_exec( _re, 
                                   _study,
                                   line.c_str(),
                                   (int) line.length(),
                                   0,
                                   0,
                                   &ovector[0],
                                   vsize );
        
        if ( PCRE_ERROR_NOMATCH == errorCode )
        {
            ovector.clear();
        }
        else
        {
            check( errorCode );
            ovector.resize( 2 * ( getCaptureCount() + 1 ) );
        }

        return Match( line, ovector );
    }


    CaptureGroup 
    Match::operator[]( size_t index ) const
    {
        CaptureGroup result = { false, INVALID_INDEX_VALUE, INVALID_INDEX_VALUE };        
        
        size_t pos = ( 2 * index );        

        if ( pos < _ovector.size() )
        {    
            result.first    = _ovector[ pos ];
            result.last     = _ovector[ pos + 1 ];
            result.matched  = INVALID_INDEX_VALUE != result.first && INVALID_INDEX_VALUE != result.last;
        }

        return result;
    }


    Match::Match( std::string const & line, std::vector<int> & ovector )
        : _str( line )
    {
        _ovector.swap( ovector );
    }
        

    std::string 
    Match::str( size_t index ) const
    {
        CaptureGroup cg = operator[]( index );

        if ( !cg.matched )
            return std::string("");        

        return _str.substr( cg.first, cg.last - cg.first );
    }


    bool 
    regexSearch( std::string const & line, Match & match, Regex const & re )
    {
        match = re.find( line );
        return match.matched();
    }


    std::string 
    regexMerge( std::string const & line, Regex const & re, char const * replacement )
    {
        assert( replacement != 0 );

        std::string result( line );

        Match match;

        if ( regexSearch( line, match, re ) )
        {
            CaptureGroup cg = match[0];
            result.replace( cg.first, cg.last - cg.first, replacement );
        }

      return result;
    }


}
}
