/**++
 *   
 *   LICENSE
 *   -------
 *   
 *   Copyright (c) 2004 Renato Mancuso
 *   All rights reserved.
 *   
 *   Redistribution and use in source and binary forms, with or without modification, are 
 *   permitted provided that the following conditions are met:
 *   
 *   - Redistributions of source code must retain the above copyright notice, this list 
 *     of conditions and the following disclaimer.
 *   
 *   - Redistributions in binary form must reproduce the above copyright notice, this list
 *     of conditions and the following disclaimer in the documentation and/or other materials 
 *     provided with the distribution.
 *   
 *   - Neither the name of Renato Mancuso nor the names of its contributors may be used to 
 *     endorse or promote products derived from this software without specific prior written 
 *     permission.
 *   
 *   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS AS IS AND ANY EXPRESS 
 *   OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY 
 *   AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR 
 *   CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL 
 *   DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, 
 *   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER 
 *   IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT 
 *   OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *   
--**/


#include "SQLineParser.hpp"
#include "ParserErrors.hpp"
#include "ParserWarnings.hpp"
#include "Regex.hpp"

#include <cstdlib>
#include <cassert>

namespace OpenEMBL
{
namespace Phoenix
{

    SQLineParser::SQLineParser(
        IParserCtx*             pParserCtx,
        IItemHandler<SQLine>*   pItemHandler)
        : ItemParserImpl<SQLine>(pParserCtx, pItemHandler)
    {
    }

    METHODIMP SQLineParser::parse()
    {
        long   theLineNumber = currentLineNumber();
        LPCSTR theLine       = currentLine();

        if (NULL == theLine)
        {
            logError(theLineNumber,
                ERROR_EOF_FOUND,
                NULL );
            return;
        }

        processLine(theLine, theLineNumber);

        nextLine();
    }

    void SQLineParser::processLine(const std::string & theLine, long theLineNumber)
    {
        //
        // Regex is: ^SQ\s{3}Sequence\s+(\d+)\s+BP\s*;\s*(\d+)\s+A\s*;\s*(\d+)\s+C\s*;\s*(\d+)\s+G\s*;\s*(\d+)\s+T\s*;\s*(\d+)\s+other\s*;\s*$
        //
        static Regex theRegex( 
            "^SQ\\s{3}Sequence\\s+(\\d+)\\s+BP\\s*;"
            "\\s*(\\d+)\\s+A\\s*;"
            "\\s*(\\d+)\\s+C\\s*;"
            "\\s*(\\d+)\\s+G\\s*;"
            "\\s*(\\d+)\\s+T\\s*;"
            "\\s*(\\d+)\\s+other\\s*;"
            "\\s*$",
            PCRE_CASELESS );    

        Match theMatches;

        if (!regexSearch(theLine, theMatches, theRegex))
        {
            logError( theLineNumber,
                      ERROR_INVALID_SQ_LINE,
                      theLine.c_str() );
            return;
        }

        SQLine theField;

        theField.SequenceLength = atoi(theMatches.str(1).c_str());
        theField.A_Count        = atoi(theMatches.str(2).c_str());
        theField.C_Count        = atoi(theMatches.str(3).c_str());
        theField.G_Count        = atoi(theMatches.str(4).c_str());
        theField.T_Count        = atoi(theMatches.str(5).c_str());
        theField.Other_Count    = atoi(theMatches.str(6).c_str());

        notifyParsed(theLineNumber, theField);
    }

}
}
