/**++
 *   
 *   LICENSE
 *   -------
 *   
 *   Copyright (c) 2004 Renato Mancuso
 *   All rights reserved.
 *   
 *   Redistribution and use in source and binary forms, with or without modification, are 
 *   permitted provided that the following conditions are met:
 *   
 *   - Redistributions of source code must retain the above copyright notice, this list 
 *     of conditions and the following disclaimer.
 *   
 *   - Redistributions in binary form must reproduce the above copyright notice, this list
 *     of conditions and the following disclaimer in the documentation and/or other materials 
 *     provided with the distribution.
 *   
 *   - Neither the name of Renato Mancuso nor the names of its contributors may be used to 
 *     endorse or promote products derived from this software without specific prior written 
 *     permission.
 *   
 *   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS AS IS AND ANY EXPRESS 
 *   OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY 
 *   AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR 
 *   CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL 
 *   DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, 
 *   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER 
 *   IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT 
 *   OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *   
--**/


#include "StringTokenizer.hpp"
#include <cstring>

namespace OpenEMBL
{
namespace Phoenix
{

    StringTokenizer::StringTokenizer(std::string const & theDelimiters,
        std::string const & theTerminals)
        : m_Delimiters(theDelimiters)
        , m_Terminals(theTerminals)
    {
        buildCharTable();
    }

    void StringTokenizer::buildCharTable()
    {
        memset(m_CharTraits, 0, sizeof(m_CharTraits));

        for (size_t i = 0; i <m_Delimiters.length(); ++i)
        {
            m_CharTraits[byte(m_Delimiters[i])] |= DELIMITER; 
        }

        for (size_t i = 0; i <m_Terminals.length(); ++i)
        {
            m_CharTraits[byte(m_Terminals[i])] |= TERMINAL; 
        }
    }

    bool StringTokenizer::isDelimiter(char c) const
    {
        return m_CharTraits[byte(c)] & DELIMITER;
    }

    bool StringTokenizer::isTerminal (char c) const
    {
        return (m_CharTraits[byte(c)] & TERMINAL) == TERMINAL;
    }

    void StringTokenizer::tokenize(
        std::string const & theLine,
        std::vector<std::string> & theTokens)
    {
        std::vector<std::string> tmp;

        std::string::const_iterator first       = theLine.begin();    
        std::string::const_iterator last        = theLine.end();      

        std::string::const_iterator token_start = first;
        std::string::const_iterator token_end   = first;

        for (; first <last; ++first, ++token_end)
        {
            if (isTerminal(*first))
            {
                if (token_end> token_start)
                    tmp.push_back(std::string(token_start, token_end));
                tmp.push_back(std::string(1,*token_end));
                token_start = token_end + 1;
            }
            else if (isDelimiter(*first))
            {
                if (token_end> token_start)
                    tmp.push_back(std::string(token_start, token_end));
                token_start = token_end + 1;                
            }
        }

        if (token_end> token_start)
            tmp.push_back(std::string(token_start, token_end));

        theTokens.swap(tmp);
    }

}
} 

