/**++
 *   
 *   LICENSE
 *   -------
 *   
 *   Copyright (c) 2004 Renato Mancuso
 *   All rights reserved.
 *   
 *   Redistribution and use in source and binary forms, with or without modification, are 
 *   permitted provided that the following conditions are met:
 *   
 *   - Redistributions of source code must retain the above copyright notice, this list 
 *     of conditions and the following disclaimer.
 *   
 *   - Redistributions in binary form must reproduce the above copyright notice, this list
 *     of conditions and the following disclaimer in the documentation and/or other materials 
 *     provided with the distribution.
 *   
 *   - Neither the name of Renato Mancuso nor the names of its contributors may be used to 
 *     endorse or promote products derived from this software without specific prior written 
 *     permission.
 *   
 *   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS AS IS AND ANY EXPRESS 
 *   OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY 
 *   AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR 
 *   CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL 
 *   DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, 
 *   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER 
 *   IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT 
 *   OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *   
--**/


#include "StringUtils.hpp"

#include <cstring>
#include <cctype>
#include <sstream>
#include <cassert>

#if !defined(_WIN32) && !defined(__WIN32__)
#   define HAS_STRINGS_HEADER 1
#endif

#if defined(__GNUC__)
#   undef HAS_STRINGS_HEADER
#   define HAS_STRINGS_HEADER 1
#endif

#ifdef HAS_STRINGS_HEADER
#   ifndef _ALL_SOURCE
#       define _ALL_SOURCE
#   endif
#   include <strings.h>
#   define _stricmp strcasecmp
#   define _strnicmp strncasecmp
#endif

#define BLANKS " \t"

namespace OpenEMBL
{
namespace Phoenix
{

    std::string trimLeft (std::string const & str)
    {
        std::string ret(str);        
        ret.erase(0, ret.find_first_not_of(BLANKS));
        return ret;
    }


    std::string trimRight(std::string const & str)
    {
        std::string ret(str);
        ret.erase(ret.find_last_not_of(BLANKS) + 1);
        return ret;
    }


    std::string trim(std::string const & str)
    {
        std::string ret(str);
        ret.erase(0, ret.find_first_not_of(BLANKS));
        ret.erase(ret.find_last_not_of(BLANKS) + 1);
        return ret;
    }


    void removeTerminator(std::string & str, char theTerminator)
    {
        if (str.empty())
            return;

        if (str[str.length() - 1] == theTerminator)
            str.erase(str.length() - 1);
    }

    int strCaseCmp (char const* left, char const* right)
    {
        assert(NULL != left);
        assert(NULL != right);
        return _stricmp(left, right);
    }

    int strNCaseCmp(char const* left, char const* right, size_t count)
    {
        assert(NULL != left);
        assert(NULL != right);
        return _strnicmp(left, right, count);
    }

    std::string makePrintable(char c, bool doQuote)
    {
        std::ostringstream oss;

        if (doQuote)     
            oss << "'";

        if (isprint(c))
            oss << c;
        else
            oss << '#' << (int) c;

        if (doQuote)     
            oss << "'";

        return oss.str();
    }

    std::string makePrintable(std::string const & str, bool doQuote)
    {
        std::ostringstream oss;

        if (doQuote)     
            oss << "'";

        for (size_t i = 0; i < str.length(); ++i)
        {
            char c = str[i];
            
            if (isprint(c))
                oss << c;
            else
                oss << '#' << (int) c;
        }

        if (doQuote)     
            oss << "'";

        return oss.str();
    }

}
}


