/**++
 *   
 *   LICENSE
 *   -------
 *   
 *   Copyright (c) 2004 Renato Mancuso
 *   All rights reserved.
 *   
 *   Redistribution and use in source and binary forms, with or without modification, are 
 *   permitted provided that the following conditions are met:
 *   
 *   - Redistributions of source code must retain the above copyright notice, this list 
 *     of conditions and the following disclaimer.
 *   
 *   - Redistributions in binary form must reproduce the above copyright notice, this list
 *     of conditions and the following disclaimer in the documentation and/or other materials 
 *     provided with the distribution.
 *   
 *   - Neither the name of Renato Mancuso nor the names of its contributors may be used to 
 *     endorse or promote products derived from this software without specific prior written 
 *     permission.
 *   
 *   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS AS IS AND ANY EXPRESS 
 *   OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY 
 *   AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR 
 *   CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL 
 *   DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, 
 *   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER 
 *   IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT 
 *   OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *   
--**/


#include "SubmissionInfoParser.hpp"
#include "ParserErrors.hpp"
#include "ParserWarnings.hpp"
#include "StringUtils.hpp"
#include "RLLineCodes.hpp"
#include "Regex.hpp"

#include <algorithm>
#include <cstdlib>
#include <cassert>

namespace OpenEMBL
{
namespace Phoenix
{

    SubmissionInfoParser::SubmissionInfoParser(
        IParserCtx*                     pParserCtx,
        IItemHandler<SubmissionInfo>*   pItemHandler)
        : ItemParserImpl<SubmissionInfo>(pParserCtx, pItemHandler)
    {
    }

    METHODIMP SubmissionInfoParser::parse()
    {
        long   theLineNumber = currentLineNumber();
        LPCSTR theLine       = currentLine();

        if (NULL == theLine)
        {
            logError(theLineNumber,
                     ERROR_EOF_FOUND,
                     NULL);
            return;
        }

        if (0 != strNCaseCmp(theLine, RL_SUBMITTED, strlen(RL_SUBMITTED)))
        {
            logError(theLineNumber,
                     ERROR_INVALID_RL_SUBMITTED_LINE,
                     theLine);
            nextLine();
            return;
        }

        StringList theLines;
        extractLines(isRLLine, theLines);
        processLines(theLines, theLineNumber);
    }

    bool SubmissionInfoParser::isRLLine(const char* theLine)
    {
        assert(NULL != theLine);
        return 0 == strncmp(theLine, "RL   ", 5);
    }

    String SubmissionInfoParser::getLineContent(const String & theLine)
    {
        assert(theLine.length() >= 5);
        return trimRight(theLine.substr(5));
    }

    void SubmissionInfoParser::processLines(const StringList & theLines, long theLineNumber)
    {
        SubmissionInfo theField;

        if (!processHeader(theLines[0], theLineNumber, theField))
            return;

        if (!processContent(theLines, theLineNumber + 1, theField))
            return;

        notifyParsed(theLineNumber, theField);
    }

    bool SubmissionInfoParser::processHeader(
        String const &      theLine, 
        long                theLineNumber, 
        SubmissionInfo &    theField)
    {
        //
        // Regex is: \(([^)]*)\)
        //
        static Regex theRegex("\\(([^)]*)\\)");

        Match theMatches;

        if (!regexSearch(theLine, theMatches, theRegex))
        {
            logError(currentLineNumber(),
                     ERROR_MISSING_DATE_IN_SUBMISSION_INFO,
                     theLine.c_str());
            return false;
        }

        theField.SubmissionDate = trim(theMatches.str(1));

        return true;
    }

    bool SubmissionInfoParser::processContent(
        StringList const &  theLines, 
        long                theLineNumber, 
        SubmissionInfo &    theField)
    {
        for (size_t i = 1; i < theLines.size(); ++i)
            theField.Info.push_back(getLineContent(theLines[i]));

        return true;
    }

}
}
