/**++
 *   
 *   LICENSE
 *   -------
 *   
 *   Copyright (c) 2004 Renato Mancuso
 *   All rights reserved.
 *   
 *   Redistribution and use in source and binary forms, with or without modification, are 
 *   permitted provided that the following conditions are met:
 *   
 *   - Redistributions of source code must retain the above copyright notice, this list 
 *     of conditions and the following disclaimer.
 *   
 *   - Redistributions in binary form must reproduce the above copyright notice, this list
 *     of conditions and the following disclaimer in the documentation and/or other materials 
 *     provided with the distribution.
 *   
 *   - Neither the name of Renato Mancuso nor the names of its contributors may be used to 
 *     endorse or promote products derived from this software without specific prior written 
 *     permission.
 *   
 *   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS AS IS AND ANY EXPRESS 
 *   OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY 
 *   AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR 
 *   CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL 
 *   DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, 
 *   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER 
 *   IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT 
 *   OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *   
--**/


#include "TaxonomyLineParser.hpp"
#include "ParserErrors.hpp"
#include "ParserWarnings.hpp"
#include "OSLineParser.hpp"
#include "OCLineParser.hpp"
#include "OGLineParser.hpp"

namespace OpenEMBL
{
namespace Phoenix
{

    TaxonomyLineParser::TaxonomyLineParser(
        IParserCtx*             pParserCtx,
        ITaxonomyHandler*       pTaxonomyHandler )
        : m_pParserCtx(pParserCtx)
        , m_pTaxonomyHandler(pTaxonomyHandler)
    {
    }

    METHODIMP_(LPCSTR) TaxonomyLineParser::currentLine() const NO_THROW
    {
        if (NULL == m_pParserCtx)
            return NULL;

        return m_pParserCtx->currentLine();
    }

    METHODIMP_(long) TaxonomyLineParser::currentLineNumber() const NO_THROW
    {
        if (NULL == m_pParserCtx)
            return TEXTSOURCE_INVALID_LINE_NUMBER;

        return m_pParserCtx->currentLineNumber();
    }

    METHODIMP_(bool) TaxonomyLineParser::nextLine()
    {
        if (NULL == m_pParserCtx)
            return false;

        return m_pParserCtx->nextLine();
    }

    METHODIMP TaxonomyLineParser::logError(long theLineNumber, int theErrorCode, char const * theMessage)
    {
        if (NULL != m_pParserCtx)
            m_pParserCtx->logError(theLineNumber, theErrorCode, theMessage);
    }

    METHODIMP TaxonomyLineParser::logWarning(long theLineNumber, int theWarning, char const * theMessage)
    {
        if (NULL != m_pParserCtx)
            m_pParserCtx->logWarning(theLineNumber, theWarning, theMessage);
    }

    bool TaxonomyLineParser::peekLine(const char* theHeader)
    {
        return NULL != currentLine() 
               &&
               0 == strncmp(currentLine(), theHeader, strlen(theHeader));
    }

    void TaxonomyLineParser::notifyBeginTaxonomy(long theLineNumber)
    {
        if (NULL != m_pTaxonomyHandler)
            m_pTaxonomyHandler->onBeginTaxonomy(theLineNumber);
    }

    void TaxonomyLineParser::notifyEndTaxonomy()
    {
        if (NULL != m_pTaxonomyHandler)
            m_pTaxonomyHandler->onEndTaxonomy();
    }

    METHODIMP TaxonomyLineParser::parse()
    {
        notifyBeginTaxonomy(currentLineNumber());

        if (NULL != currentLine())
        {
            OSLineParser theParser(this, m_pTaxonomyHandler);
            theParser.parse();
        }

        if (peekLine("OC   "))
        {
            OCLineParser theParser(this, m_pTaxonomyHandler);                
            theParser.parse();
        }
        else
        {
            logWarning(currentLineNumber(),
                       WARNING_MISSING_OC_LINE,
                       currentLine());
        }

        if (peekLine("OG   "))
        {
            OGLineParser theParser(this, m_pTaxonomyHandler);
            theParser.parse();
        }

        notifyEndTaxonomy();            
    }   

}
}
