/**++
 *   
 *   LICENSE
 *   -------
 *   
 *   Copyright (c) 2004 Renato Mancuso
 *   All rights reserved.
 *   
 *   Redistribution and use in source and binary forms, with or without modification, are 
 *   permitted provided that the following conditions are met:
 *   
 *   - Redistributions of source code must retain the above copyright notice, this list 
 *     of conditions and the following disclaimer.
 *   
 *   - Redistributions in binary form must reproduce the above copyright notice, this list
 *     of conditions and the following disclaimer in the documentation and/or other materials 
 *     provided with the distribution.
 *   
 *   - Neither the name of Renato Mancuso nor the names of its contributors may be used to 
 *     endorse or promote products derived from this software without specific prior written 
 *     permission.
 *   
 *   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS AS IS AND ANY EXPRESS 
 *   OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY 
 *   AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR 
 *   CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL 
 *   DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, 
 *   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER 
 *   IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT 
 *   OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *   
--**/


#include "TextFileImpl.hpp"

#include <cerrno>
#include <cstring>
#include <cassert>

#if defined(_MSC_VER)
#   pragma warning(disable: 4290) // C++ exception specification ignored except to indicate a function is not __declspec(nothrow)
#endif

namespace OpenEMBL
{
namespace Phoenix
{

    TextFileImpl::TextFileImpl(char const* theFilename) throw(OSError)
        : m_hFile      (NULL)
        , m_LineNumber (0)   
        , m_EOF        (false)
    {
        assert(NULL != theFilename);

        m_cchLine[0]  = 0;
        m_hFile       = fopen(theFilename, "r");

        if (NULL == m_hFile)
            throw OSError(errno);

        setvbuf(m_hFile, m_IOBuf, _IOFBF, sizeof(m_IOBuf)/sizeof(char));
        nextLine();
    }


    TextFileImpl::~TextFileImpl() throw()
    {
        fclose(m_hFile);
    }


    char const* TextFileImpl::currentLine() const throw()
    {
        if (m_EOF)
            return NULL;

        return m_cchLine;
    }


    long TextFileImpl::currentLineNumber() const throw()
    {
        return m_LineNumber;
    }


    bool TextFileImpl::nextLine() throw(OSError, LineTooLong)
    {      
        if (!m_EOF)
        {                    
            ++m_LineNumber;
            m_EOF = NULL == fgets(m_cchLine, sizeof(m_cchLine), m_hFile);            
        }

        if (m_EOF)
        {
            if (!feof(m_hFile))
                throw OSError(errno); 
        }
        else
        {
            size_t slen = strlen(m_cchLine);

            //
            // remove terminal newline character from line
            //
            if ((slen> 0) && ('\n' == m_cchLine[slen - 1]))
                m_cchLine[slen - 1] = 0;
            else 
            {
                throw LineTooLong(m_LineNumber);
            }
        }

        return !m_EOF;
    }


} 
}

