/**++
 *   
 *   LICENSE
 *   -------
 *   
 *   Copyright (c) 2004 Renato Mancuso
 *   All rights reserved.
 *   
 *   Redistribution and use in source and binary forms, with or without modification, are 
 *   permitted provided that the following conditions are met:
 *   
 *   - Redistributions of source code must retain the above copyright notice, this list 
 *     of conditions and the following disclaimer.
 *   
 *   - Redistributions in binary form must reproduce the above copyright notice, this list
 *     of conditions and the following disclaimer in the documentation and/or other materials 
 *     provided with the distribution.
 *   
 *   - Neither the name of Renato Mancuso nor the names of its contributors may be used to 
 *     endorse or promote products derived from this software without specific prior written 
 *     permission.
 *   
 *   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS AS IS AND ANY EXPRESS 
 *   OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY 
 *   AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR 
 *   CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL 
 *   DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, 
 *   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER 
 *   IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT 
 *   OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *   
--**/


/*****************************************************************************
 * 
 *  Utility to display on screen all the events that the parser generates
 *  while reading a flat file.
 * 
 ****************************************************************************/

#include <cstdio>
#include <iostream>
#include <Phoenix.hpp>

#if defined(_MSC_VER)
#   pragma warning(disable: 4267)
#endif

using namespace OpenEMBL::Phoenix;
using namespace std;

namespace {

    void showBeginEvent(char const * theEventName, long theLineNumber)
    {
        cerr << "\n==> " << theEventName << " (line: " << theLineNumber << ")\n";
    }

    void showEndEvent(char const * theEventName)
    {
        cerr << "\n<== " << theEventName << "\n";
    }

    void showEvent(char const * theEventName, long theLineNumber)
    {
        cerr << "\n* " << theEventName << " (line: " << theLineNumber << ")\n";
    }

    void showFieldValue(char const * theFieldName, int theFieldValue)
    {
        cerr << "    " << theFieldName << ": " << theFieldValue << "\n";
    }

    void showFieldValue(char const * theFieldName, string const & theFieldValue)
    {
        cerr << "    " << theFieldName << ": " << theFieldValue << "\n";
    }

    void showFieldValue(char const * theFieldName, bool theFieldValue)
    {
        cerr << "    " << theFieldName << ": " << (theFieldValue ? "true" : "false") << "\n";
    }
    
    void showFieldValue(char const * theFieldName, long theFieldValue)
    {
        cerr << "    " << theFieldName << ": " << theFieldValue << "\n";
    }

    void showFieldValue(char const * theFieldName, vector<string> const & theFieldValue)
    {
        cerr << "    " << theFieldName << "[" << theFieldValue.size() << "]:\n";

        if (theFieldValue.empty())
            cerr << "        <EMPTY>\n";
        else
        {
            for (size_t i = 0; i < theFieldValue.size(); ++i)
                cerr << "        [" << i << "] " << theFieldValue[i] << "\n";

        }
    }

    void showFieldValue(char const * theFieldName, OpenEMBL::Phoenix::Range theFieldValue)
    {
        cerr << "    " << theFieldName << ": " << theFieldValue.First << "-" << theFieldValue.Last << "\n";
    }

    void showFieldValue(char const * theFieldName, vector<Range> const & theFieldValue)
    {
        cerr << "    " << theFieldName << "[" << theFieldValue.size() << "]:\n";

        if (theFieldValue.empty())
            cerr << "        <EMPTY>\n";
        else
        {
            for (size_t i = 0; i < theFieldValue.size(); ++i)
                cerr << "        [" << i << "] " 
                     << theFieldValue[i].First << "-" << theFieldValue[i].Last << "\n";

        }
    }

    void showFieldValue(char const * theFieldName, vector<LocationElement> const & theElements)
    {
        cerr << "    " << theFieldName << "[" << theElements.size() << "]:\n";

        if (theElements.empty())
            cerr << "          <EMPTY>\n";
        else
        {
            for (size_t i = 0; i < theElements.size(); ++i)
            {
                LocationElement const & x = theElements[i];
                cerr << "          [" << i << "]\n";
                cerr << "          <RECORD START>\n";
                cerr << "          AccessionNumber: " << x.AccessionNumber << "\n";
                cerr << "          SequenceVersion: " << x.SequenceVersion << "\n";
                cerr << "          e_flags        : " << hex << "0x" << x.e_flags << dec << "\n";
                cerr << "          x_flags        : " << hex << "0x" << x.x_flags << dec << "\n";
                cerr << "          y_flags        : " << hex << "0x" << x.y_flags << dec << "\n";
                cerr << "          x1, x2         : " << x.x1 << ", " << x.x2 << "\n";
                cerr << "          y1, y2         : " << x.y1 << ", " << x.y2 << "\n";
                cerr << "          <RECORD END>\n";
            }
        }
    }

} // namespace

#define BEGIN_EVENT(ev, line) showBeginEvent(ev, line)

#define END_EVENT(ev) showEndEvent(ev)

#define SHOW_EVENT(ev, line) showEvent(ev, line)

#define SHOW_FIELD_VALUE(field) showFieldValue(#field, field)

class PARSER 
    : public FileParserHost
{
private:

    typedef FileParserHost inherited;


public:
    PARSER(char const * theFilename)
        : FileParserHost( theFilename )
    {
    }

    // Entry
    METHOD onBeginEntry(long theLineNumber)
    {
        FileParserHost::onBeginEntry(theLineNumber);
        BEGIN_EVENT("*** onBeginEntry ***", theLineNumber);
    }

    METHOD onEndEntry()
    {
        FileParserHost::onEndEntry();
        END_EVENT("*** onEndEntry ***");
    }

    // Individual Fields
    METHOD onID(long theLineNumber, IDLine const & theField)
    {
        SHOW_EVENT("onID", theLineNumber);
        SHOW_FIELD_VALUE(theField.EntryName);
        SHOW_FIELD_VALUE(theField.DataClass);
        SHOW_FIELD_VALUE(theField.Circular);
        SHOW_FIELD_VALUE(theField.MolType);
        SHOW_FIELD_VALUE(theField.Division);
        SHOW_FIELD_VALUE(theField.SequenceLength);
		SHOW_FIELD_VALUE(theField.SequenceVersion);
    }

    METHOD onAC(long theLineNumber, ACLine const & theField)
    {
        SHOW_EVENT("onAC", theLineNumber);
        SHOW_FIELD_VALUE(theField.AccessionNumberList);
    }

    METHOD onSV(long theLineNumber, SVLine const & theField)
    {
        SHOW_EVENT("onSV", theLineNumber);
        SHOW_FIELD_VALUE(theField.AccessionNumber);        
        SHOW_FIELD_VALUE(theField.Version);
    }
   
    METHOD onDT(long theLineNumber, DTLine const & theField)
    {
        SHOW_EVENT("onDT", theLineNumber);
        SHOW_FIELD_VALUE(theField.FirstCreated);
        SHOW_FIELD_VALUE(theField.ReleaseCreated);
        SHOW_FIELD_VALUE(theField.LastUpdated);
        SHOW_FIELD_VALUE(theField.ReleaseLastUpdated);
        SHOW_FIELD_VALUE(theField.ExternalVersion);
    }

    METHOD onXX(long theLineNumber, XXLine const & theField)
    {
        SHOW_EVENT("onXX", theLineNumber);
    }
    
    METHOD onDE(long theLineNumber, DELine const & theField)
    {
        SHOW_EVENT("onDE", theLineNumber);
        SHOW_FIELD_VALUE(theField.Text);
    }
    
    METHOD onKW(long theLineNumber, KWLine const & theField)
    {
        SHOW_EVENT("onKW", theLineNumber);
        SHOW_FIELD_VALUE(theField.Keywords);
    }
    
    METHOD onDR(long theLineNumber, DRLine const & theField)
    {
        SHOW_EVENT("onDR", theLineNumber);
        SHOW_FIELD_VALUE(theField.DatabaseIdentifier);
        SHOW_FIELD_VALUE(theField.PrimaryIdentifier);
        SHOW_FIELD_VALUE(theField.SecondaryIndentifier);
    }

    METHOD onCC(long theLineNumber, CCLine const & theField)
    {
        SHOW_EVENT("onCC", theLineNumber);
        SHOW_FIELD_VALUE(theField.Comment);
    }

    METHOD onSQ(long theLineNumber, SQLine const & theField)
    {
        SHOW_EVENT("onSQ", theLineNumber);
        SHOW_FIELD_VALUE(theField.A_Count);
        SHOW_FIELD_VALUE(theField.C_Count);
        SHOW_FIELD_VALUE(theField.G_Count);
        SHOW_FIELD_VALUE(theField.T_Count);
        SHOW_FIELD_VALUE(theField.Other_Count);
        SHOW_FIELD_VALUE(theField.SequenceLength);
    }    

    METHOD onBeginTaxonomyLines(long theLineNumber)
    {
        BEGIN_EVENT("onBeginTaxonomyLines", theLineNumber);
    }

    METHOD onOS(long theLineNumber, OSLine const & theField)
    {
        SHOW_EVENT("onOS", theLineNumber);
        SHOW_FIELD_VALUE(theField.Organism);
    }

    METHOD onOC(long theLineNumber, OCLine const & theField)
    {
        SHOW_EVENT("onOC", theLineNumber);
        SHOW_FIELD_VALUE(theField.Lineage);
    }

    METHOD onOG(long theLineNumber, OGLine const & theField)
    {
        SHOW_EVENT("onOG", theLineNumber);
        SHOW_FIELD_VALUE(theField.Organelles);
    }
    
    METHOD onEndTaxonomyLines()
    {
        END_EVENT("onEndTaxonomyLines");
    }
    
    METHOD onBeginFeatureTable(long theLineNumber)
    {
        BEGIN_EVENT("onBeginFeatureTable", theLineNumber);
    }

    METHOD onEndFeatureTable()
    {
        END_EVENT("onEndFeatureTable");
    }

    METHOD onBeginFeatureKey(long theLineNumber, FeatureKey const & theField)
    {
        BEGIN_EVENT("onBeginFeatureKey", theLineNumber);
        SHOW_FIELD_VALUE(theField.Key);
        SHOW_FIELD_VALUE(theField.Location.IsJoin);
        SHOW_FIELD_VALUE(theField.Location.IsComplement);
        SHOW_FIELD_VALUE(theField.Location.LocationElements);
    }

    METHOD onFeatureQualifier(long theLineNumber, FeatureQualifier const  & theQualifier)
    {
        SHOW_EVENT("onFeatureQualifier", theLineNumber);
        SHOW_FIELD_VALUE(theQualifier.Name);
        SHOW_FIELD_VALUE(theQualifier.Value);
    }

    METHOD onEndFeatureKey()
    {
        END_EVENT("onEndFeatureKey");
    }

    METHOD onBeginSequenceData(long theLineNumber)
    {
        BEGIN_EVENT("onBeginSequenceData", theLineNumber);
    }

    METHOD onSequenceLine(long theLineNumber, SequenceLine const & theField)
    {
        SHOW_EVENT("onSequenceLine", theLineNumber);
        SHOW_FIELD_VALUE(theField.Sequence);
    }

    METHOD onEndSequenceData()
    {
        END_EVENT("onEndSequenceData");
    }

    METHOD onBeginTPALines(long theLineNumber)
    {
        BEGIN_EVENT("onBeginTPALines", theLineNumber);
    }

    METHOD onAS(long theLineNumber, ASLine const & theField)
    {
        SHOW_EVENT("onAS", theLineNumber);
        SHOW_FIELD_VALUE(theField.TPASpan);
        SHOW_FIELD_VALUE(theField.PrimaryIdentifier);
        SHOW_FIELD_VALUE(theField.PrimarySpan);
        SHOW_FIELD_VALUE(theField.Complement);
    }

    METHOD onEndTPALines()
    {
        END_EVENT("onEndTPALines");
    }

    METHOD onBeginCOLines(long theLineNumber, bool isComplement)
    {
        if (isComplement)
            BEGIN_EVENT("onBeginCOLines (isComplement = true)", theLineNumber);
        else
            BEGIN_EVENT("onBeginCOLines (isComplement = false)", theLineNumber);
    }

    METHOD onCOSegmentInfo (long theLineNumber, COSegmentInfo const & theSegmentInfo)
    {
        SHOW_EVENT("onCOSegmentInfo", theLineNumber);
        SHOW_FIELD_VALUE(theSegmentInfo.AccessionNumber);
        SHOW_FIELD_VALUE(theSegmentInfo.SequenceVersion);
        SHOW_FIELD_VALUE(theSegmentInfo.Complement);
        SHOW_FIELD_VALUE(theSegmentInfo.Span);
    }

    METHOD onCOGapInfo (long theLineNumber, COGapInfo const & theGapInfo)
    {
        SHOW_EVENT("onCOGapInfo", theLineNumber);
        SHOW_FIELD_VALUE(theGapInfo.Length);
    }

    METHOD onEndCOLines()
    {
        END_EVENT("onEndCOLines");     
    }

    METHOD onACStar(long theLineNumber, ACStarLine const & theField)
    {
        SHOW_EVENT("onACStar", theLineNumber);
        SHOW_FIELD_VALUE(theField.GenomeProjectID);
        SHOW_FIELD_VALUE(theField.Version);
        SHOW_FIELD_VALUE(theField.WGSVersion);
    }

    METHOD onBeginPublication(long theLineNumber)
    {
        BEGIN_EVENT("onBeginPublication", theLineNumber);
    }

    METHOD onRN(long theLineNumber, RNLine const & theField)
    {
        SHOW_EVENT("onRN", theLineNumber);
        SHOW_FIELD_VALUE(theField.ReferenceNumber);
    }

    METHOD onRC(long theLineNumber, RCLine const & theField)
    {
        SHOW_EVENT("onRC", theLineNumber);
        SHOW_FIELD_VALUE(theField.Comment);
    }

    METHOD onRP(long theLineNumber, RPLine const & theField)
    {
        SHOW_EVENT("onRP", theLineNumber);
        SHOW_FIELD_VALUE(theField.Positions);
    }

    METHOD onRX(long theLineNumber, RXLine const & theField)
    {
        SHOW_EVENT("onRX", theLineNumber);
        SHOW_FIELD_VALUE(theField.DatabaseIdentifier);
        SHOW_FIELD_VALUE(theField.Identifier);
    }

    METHOD onRG(long theLineNumber, RGLine const & theField)
    {
        SHOW_EVENT("onRG", theLineNumber);
        SHOW_FIELD_VALUE(theField.Consortium);
    }

    METHOD onRA(long theLineNumber, RALine const & theField)
    {
        SHOW_EVENT("onRA", theLineNumber);
        SHOW_FIELD_VALUE(theField.Authors);
    }

    METHOD onRT(long theLineNumber, RTLine const & theField)
    {
        SHOW_EVENT("onRT", theLineNumber);
        SHOW_FIELD_VALUE(theField.Text);
    }

    METHOD onSubmission        (long theLineNumber, SubmissionInfo         const & theField)
    {
        SHOW_EVENT("onSubmission", theLineNumber);
        SHOW_FIELD_VALUE(theField.SubmissionDate);
        SHOW_FIELD_VALUE(theField.Info);
    }

    METHOD onUnpublished       (long theLineNumber, UnpublishedInfo        const & theField)
    {
        SHOW_EVENT("onUnpublished", theLineNumber);
    }

    METHOD onBook              (long theLineNumber, BookInfo               const & theField)
    {
        SHOW_EVENT("onBook", theLineNumber);
        SHOW_FIELD_VALUE(theField.Title);
        SHOW_FIELD_VALUE(theField.PublicationDate);
        SHOW_FIELD_VALUE(theField.FirstPage);
        SHOW_FIELD_VALUE(theField.LastPage);
        SHOW_FIELD_VALUE(theField.Publisher);
        SHOW_FIELD_VALUE(theField.Editors);
    }

    METHOD onThesis            (long theLineNumber, ThesisInfo             const & theField)
    {
        SHOW_EVENT("onThesis", theLineNumber);
        SHOW_FIELD_VALUE(theField.Institute);
        SHOW_FIELD_VALUE(theField.Year);
    }

    METHOD onPatent            (long theLineNumber, PatentInfo             const & theField)
    {
        SHOW_EVENT("onPatent", theLineNumber);
        SHOW_FIELD_VALUE(theField.PatentNumber);
        SHOW_FIELD_VALUE(theField.Date);
        SHOW_FIELD_VALUE(theField.Applicants);
    }

    METHOD onOnlineJournal     (long theLineNumber, OnlineJournalInfo      const & theField)
    {
        SHOW_EVENT("onOnlineJournal", theLineNumber);
    }

    METHOD onElectronicResource(long theLineNumber, ElectronicResourceInfo const & theField)
    {
        SHOW_EVENT("onElectronicResource", theLineNumber);
        SHOW_FIELD_VALUE(theField.FreeText);
    }

	METHOD onMiscResource(long theLineNumber, MiscResourceInfo const & theField)
    {
        SHOW_EVENT("onMiscResource", theLineNumber);
        SHOW_FIELD_VALUE(theField.FreeText);
    }

    METHOD onJournalArticle    (long theLineNumber, JournalArticleInfo     const & theField)
    {
        SHOW_EVENT("onJournalArticle", theLineNumber);
        SHOW_FIELD_VALUE(theField.JournalName);
        SHOW_FIELD_VALUE(theField.Volume);
        SHOW_FIELD_VALUE(theField.Issue);
        SHOW_FIELD_VALUE(theField.FirstPage);
        SHOW_FIELD_VALUE(theField.LastPage);
        SHOW_FIELD_VALUE(theField.PublicationDate);
    }

    METHOD onEndPublication()
    {
        END_EVENT("onEndPublication");
    }    
};

int main(int argc, char* argv[])
{
    using namespace OpenEMBL;

    if (argc < 2)
        return 1;

    PARSER theParser(argv[1]);
    theParser.parse();

    return 0;
}

