/**++
 *   
 *   LICENSE
 *   -------
 *   
 *   Copyright (c) 2004 Renato Mancuso
 *   All rights reserved.
 *   
 *   Redistribution and use in source and binary forms, with or without modification, are 
 *   permitted provided that the following conditions are met:
 *   
 *   - Redistributions of source code must retain the above copyright notice, this list 
 *     of conditions and the following disclaimer.
 *   
 *   - Redistributions in binary form must reproduce the above copyright notice, this list
 *     of conditions and the following disclaimer in the documentation and/or other materials 
 *     provided with the distribution.
 *   
 *   - Neither the name of Renato Mancuso nor the names of its contributors may be used to 
 *     endorse or promote products derived from this software without specific prior written 
 *     permission.
 *   
 *   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS AS IS AND ANY EXPRESS 
 *   OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY 
 *   AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR 
 *   CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL 
 *   DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, 
 *   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER 
 *   IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT 
 *   OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *   
--**/


/*****************************************************************************
 * 
 *  A moronic flat file to XML converter.
 *
 *  Please see the ff2xml sample application for a more realistic flat file 
 *  to XML conversion example.
 * 
 ****************************************************************************/

#include <cstdio>
#include <iostream>
#include <sstream>
#include <Phoenix.hpp>

using namespace OpenEMBL::Phoenix;
using namespace std;

class XmlWriter
{
public:
    explicit XmlWriter(ostream & output)
        : _output(output)
    {
        _output << "<?xml version=\"1.0\" ?>\n";
    }

    ~XmlWriter()
    {
        _output << flush;
    }

    void writeBeginTag(char const * tag)
    {
        _output << "<" << tag << ">\n";
    }

    void writeBeginTag(char const * tag, char const * attributes)
    {
        _output << "<" << tag << " " << attributes << " >\n";
    }

    void writeEndTag(char const * tag)
    {
         _output << "</" << tag << ">\n";
    }

    void writeLine(char const* str)
    {
        static const char LT[]        = "&lt;";
        static const char GT[]        = "&gt;";
        static const char AMPERSAND[] = "&amp;";

        const char* first = str;
        const char* last  = str + strlen(str);

        for (; first < last; ++first)
        {
            switch (*first)
            {
            case '<':
                _output << LT;
                break;

            case '>':
                _output << GT;
                break;

            case '&':
                _output << AMPERSAND;
                break;

            default:
                _output.put(*first);
            }
        }

        _output << "\n";
    }

    void writeElement(char const * elementName, char const * elementValue)
    {
        writeBeginTag(elementName);
        writeLine(elementValue);
        writeEndTag(elementName);
    }

    void writeElement(char const * elementName, std::string const & elementValue)
    {
        writeBeginTag(elementName);
        writeLine(elementValue.c_str());
        writeEndTag(elementName);
    }

    void writeElement(char const * elementName, StringList const & elementValue)
    {
        writeBeginTag(elementName);
        
        for (size_t i = 0; i < elementValue.size(); ++i)
            writeLine(elementValue[i].c_str());

        writeEndTag(elementName);
    }

    void writeElement(char const * elementName, long elementValue)
    {
        writeBeginTag(elementName);
        _output << elementValue << "\n";
        writeEndTag(elementName);
    }

    void writeElement(char const * elementName, int elementValue)
    {
        writeBeginTag(elementName);
        _output << elementValue << "\n";
        writeEndTag(elementName);
    }

    void writeElement(char const * elementName, bool elementValue)
    {
        writeBeginTag(elementName);

        if (elementValue)
            _output << "TRUE" << "\n";
        else
            _output << "FALSE" << "\n";
        
        writeEndTag(elementName);
    }

    void writeElement(char const * elementName, Range const & elementValue)
    {
        writeBeginTag(elementName);
        _output << elementValue.First << "-" << elementValue.Last << "\n";        
        writeEndTag(elementName);
    }

    void writeElement(char const * elementName, char const * itemName, StringList const & values)
    {
        writeBeginTag(elementName);               

        for (size_t i = 0; i < values.size(); ++i)
            writeElement(itemName, values[i].c_str());

        writeEndTag(elementName);
    }

    void writeElement(char const * elementName, char const * itemName, vector<Range> const & values)
    {
        writeBeginTag(elementName);               

        for (size_t i = 0; i < values.size(); ++i)
            writeElement(itemName, values[i]);

        writeEndTag(elementName);
    }

private:
    ostream & _output;
};


class PARSER 
    : public FileParserHost
{
public:
    PARSER(char const * theFilename)
        : FileParserHost(theFilename)
        , xml(cout)
    {
        xml.writeBeginTag("EntryArchive");
    }

    ~PARSER() throw()
    {
        xml.writeEndTag("EntryArchive");
    }

    METHOD onBeginEntry(long theLineNumber)
    {
        FileParserHost::onBeginEntry(theLineNumber);
        xml.writeBeginTag("Entry");
    }

    METHOD onEndEntry()
    {
        FileParserHost::onEndEntry();
        xml.writeEndTag("Entry");
    }

    METHOD onID(long theLineNumber, IDLine const & theField)
    {
        xml.writeBeginTag("ID");
        
        xml.writeElement("EntryName", theField.EntryName);
        xml.writeElement("DataClass", theField.DataClass);

        if (theField.Circular)
            xml.writeElement("Circular", theField.Circular);

        xml.writeElement("MolType", theField.MolType);
        xml.writeElement("Division", theField.Division);
        xml.writeElement("SequenceLength", theField.SequenceLength);
		xml.writeElement("SequenceVersion", theField.SequenceVersion);
        
        xml.writeEndTag("ID");
    }

    METHOD onAC(long theLineNumber, ACLine const & theField)
    {
        if (theField.AccessionNumberList.empty())
            return;

        xml.writeElement("AC", "AccessionNumber", theField.AccessionNumberList);
    }

    METHOD onSV(long theLineNumber, SVLine const & theField)
    {
        xml.writeBeginTag("SV");
        xml.writeElement("AccessionNumber", theField.AccessionNumber);
        xml.writeElement("SequenceVersion", theField.Version);
        xml.writeEndTag("SV");
    }

    METHOD onDT(long theLineNumber, DTLine const & theField)
    {
        xml.writeBeginTag("DT");

        xml.writeElement("FirstCreated", theField.FirstCreated);
        xml.writeElement("ReleaseCreated", theField.ReleaseCreated);
        xml.writeElement("LastUpdated", theField.LastUpdated);
        xml.writeElement("ReleaseLastUpdated", theField.ReleaseLastUpdated);
        xml.writeElement("EntryVersion", theField.ExternalVersion);

        xml.writeEndTag("DT");
    }

    METHOD onDE(long theLineNumber, DELine const & theField)
    {
        xml.writeElement("DE", theField.Text);
    }

    METHOD onKW(long theLineNumber, KWLine const & theField)
    {
        if (theField.Keywords.empty())
            return;

        if (theField.Keywords.size() == 1 && theField.Keywords[0].empty())
            return;

        xml.writeElement("KW", "Keyword", theField.Keywords);
    }

    METHOD onDR(long theLineNumber, DRLine const & theField)
    {
        xml.writeBeginTag("DR");

        xml.writeElement("DatabaseIdentifier", theField.DatabaseIdentifier);
        xml.writeElement("PrimaryIdentifier", theField.PrimaryIdentifier);

        if (!theField.SecondaryIndentifier.empty())
            xml.writeElement("SecondaryIdentifier", theField.SecondaryIndentifier);

        xml.writeEndTag("DR");
    }

    METHOD onCC(long theLineNumber, CCLine const & theField)
    {
        xml.writeElement("CC", theField.Comment);
    }

    METHOD onSQ(long theLineNumber, SQLine const & theField)
    {
        xml.writeBeginTag("SQ");

        xml.writeElement("A_Count", theField.A_Count);
        xml.writeElement("T_Count", theField.T_Count);
        xml.writeElement("C_Count", theField.C_Count);
        xml.writeElement("G_Count", theField.G_Count);
        xml.writeElement("Other_Count", theField.Other_Count);
        xml.writeElement("SequenceLength", theField.SequenceLength);

        xml.writeEndTag("SQ");
    }

    METHOD onXX(long theLineNumber, XXLine const & theField)
    {
        // NULL;
    }

    METHOD onBeginTaxonomyLines(long theLineNumber)
    {
        xml.writeBeginTag("TaxonomyInfo");
    }

    METHOD onOS(long theLineNumber, OSLine const & theField)
    {
        xml.writeElement("OS", theField.Organism);
    }

    METHOD onOC(long theLineNumber, OCLine const & theField)
    {
        xml.writeElement("OC", "Taxon", theField.Lineage);
    }

    METHOD onOG(long theLineNumber, OGLine const & theField)
    {
        xml.writeElement("OG", "Organelle", theField.Organelles);
    }

    METHOD onEndTaxonomyLines()
    {
        xml.writeEndTag("TaxonomyInfo");
    }

    METHOD onBeginFeatureTable(long theLineNumber)
    {
        xml.writeBeginTag("FeatureTable");
    }

    void writeLocation(FeatureLocation const & theLocation)
    {
        xml.writeBeginTag("Location");

        if (theLocation.IsComplement)
            xml.writeElement("Complement", theLocation.IsComplement);
        
        if (theLocation.LocationElements.size() > 1)
        {
            if (theLocation.IsJoin)
                xml.writeElement("JoinType", "Join");
            else
                xml.writeElement("JoinType", "Order");
        }

        writeLocationElementList(theLocation.LocationElements);

        xml.writeEndTag("Location");
    }

    void writeLocationElementList(LocationElementList const & theElements)
    {
        for (size_t i = 0; i < theElements.size(); ++i)
        {
            LocationElement const & le = theElements[i];            
            writeLocationElement(le);
        }
    }

    void writeLocationElement(LocationElement const & theElement)
    {
        xml.writeBeginTag("LocationElement");

        if (theElement.e_flags & LOCATION_ELEMENT_COMPLEMENT)
            xml.writeElement("Complement", true);

        if (theElement.AccessionNumber.length() > 0)
        {
            xml.writeElement("AccessionNumber", theElement.AccessionNumber);
            xml.writeElement("SequenceVersion", theElement.SequenceVersion);
        }

        switch(theElement.e_flags & 0xf)
        {
        case LOCATION_ELEMENT_SINGLE:
            xml.writeElement("Kind", "x");
            writeBasePosition(theElement.x1, theElement.x2, theElement.x_flags);
            break;

        case LOCATION_ELEMENT_INFRABASE:
            xml.writeElement("Kind", "x^y");
            writeBasePosition(theElement.x1, theElement.x2, theElement.x_flags);
            writeBasePosition(theElement.y1, theElement.y2, theElement.y_flags);
            break;                       

        case LOCATION_ELEMENT_RANGE:
            xml.writeElement("Kind", "x..y");
            writeBasePosition(theElement.x1, theElement.x2, theElement.x_flags);
            writeBasePosition(theElement.y1, theElement.y2, theElement.y_flags);
            break;
        }

        xml.writeEndTag("LocationElement");
    }

    void writeBasePosition(long x1, long x2, unsigned flags)
    {
        ostringstream oss;           

        switch (flags)
        {
        case BASE_POSITION_SIMPLE_BASE:            
            xml.writeBeginTag("BasePosition");
            oss << x1;            
            break;

        case BASE_POSITION_LEFT_OPEN:
            xml.writeBeginTag("BasePosition", "kind=\"left open\"");
            oss << x1;
            break;

        case BASE_POSITION_RIGHT_OPEN:
            xml.writeBeginTag("BasePosition", "kind=\"right open\"");
            oss << x1;
            break;

        case BASE_POSITION_SPAN:
            xml.writeBeginTag("BasePosition", "kind=\"fuzzy\"");
            oss << "(" << x1 << "." << x2 << ")";
            break;

        default:
            xml.writeBeginTag("BasePosition");
        }

        xml.writeLine(oss.str().c_str());

        xml.writeEndTag("BasePosition");
    }

    METHOD onBeginFeatureKey(long theLineNumber, FeatureKey const & theField)
    {
        xml.writeBeginTag("Key");
        xml.writeElement("Name", theField.Key);
        writeLocation(theField.Location);
    }

    METHOD onFeatureQualifier(long theLineNumber, FeatureQualifier const  & theQualifier)
    {
        xml.writeBeginTag("Qualifier");

        xml.writeElement("Name", theQualifier.Name);
        xml.writeElement("Value", theQualifier.Value);

        xml.writeEndTag("Qualifier");
    }

    METHOD onEndFeatureKey()
    {
        xml.writeEndTag("Key");
    }

    METHOD onEndFeatureTable()
    {
        xml.writeEndTag("FeatureTable");
    }

    METHOD onBeginSequenceData(long theLineNumber)
    {
        xml.writeBeginTag("Sequence");
    }

    METHOD onSequenceLine(long theLineNumber, SequenceLine const & theField)
    {
        xml.writeElement("SequenceLine", theField.Sequence);
    }

    METHOD onEndSequenceData()
    {
        xml.writeEndTag("Sequence");
    }

    METHOD onBeginTPALines(long theLineNumber)
    {
        xml.writeBeginTag("AssemblyInfo");
    }

    METHOD onAS(long theLineNumber, ASLine const & theField)
    {
        xml.writeBeginTag("AS");

        xml.writeElement("TpaSpan", theField.TPASpan);
        xml.writeElement("PrimaryIdentifier", theField.PrimaryIdentifier);
        xml.writeElement("PrimarySpan", theField.PrimarySpan);
        xml.writeElement("Complement", theField.Complement);

        xml.writeEndTag("AS");
    }

    METHOD onEndTPALines()
    {
        xml.writeEndTag("AssemblyInfo");
    }

    METHOD onBeginCOLines(long theLineNumber, bool isComplement)
    {
        xml.writeBeginTag("ContigInfo");
        xml.writeElement("Complement", isComplement);        
    }

    METHOD onCOSegmentInfo (long theLineNumber, COSegmentInfo const & theSegmentInfo)
    {
        xml.writeBeginTag("Segment");

        xml.writeElement("AccessionNumber", theSegmentInfo.AccessionNumber);
        xml.writeElement("SequenceVersion", theSegmentInfo.SequenceVersion);
        xml.writeElement("Complement", theSegmentInfo.Complement);
        xml.writeElement("Span", theSegmentInfo.Span);

        xml.writeEndTag("Segment");
    }

    METHOD onCOGapInfo (long theLineNumber, COGapInfo const & theGapInfo)
    {
        xml.writeBeginTag("Gap");

        if (theGapInfo.Length > 0)
        {
            ostringstream oss;
            oss << theGapInfo.Length;

            xml.writeLine(oss.str().c_str());
        }

        xml.writeEndTag("Gap");
    }

    METHOD onEndCOLines()
    {
        xml.writeEndTag("ContigInfo");
    }

    METHOD onACStar(long theLineNumber, ACStarLine const & theField)
    {
        xml.writeBeginTag("GenomeProjectInfo");

        xml.writeElement("GpID", theField.GenomeProjectID);

        if (theField.Version > 0)
            xml.writeElement("GpVersion", theField.Version);

        if (theField.WGSVersion > 0)
            xml.writeElement("WgsVersion", theField.WGSVersion);

        xml.writeEndTag("GenomeProjectInfo");
    }

    METHOD onBeginPublication(long theLineNumber)
    {
        xml.writeBeginTag("Publication");
    }

    METHOD onRN(long theLineNumber, RNLine const & theField)
    {
        xml.writeElement("RN", theField.ReferenceNumber);
    }

    METHOD onRC(long theLineNumber, RCLine const & theField)
    {
        xml.writeElement("RC", theField.Comment);
    }

    METHOD onRP(long theLineNumber, RPLine const & theField)
    {
        xml.writeElement("RP", "Range", theField.Positions);
    }

    METHOD onRX(long theLineNumber, RXLine const & theField)
    {
        xml.writeBeginTag("RX");

        xml.writeElement("Database", theField.DatabaseIdentifier);
        xml.writeElement("Identifier", theField.Identifier);

        xml.writeEndTag("RX");
    }

    METHOD onRG(long theLineNumber, RGLine const & theField)
    {
        xml.writeElement("RG", theField.Consortium);
    }

    METHOD onRA(long theLineNumber, RALine const & theField)
    {
        xml.writeElement("RA", "Author", theField.Authors);
    }

    METHOD onRT(long theLineNumber, RTLine const & theField)
    {
        if (theField.Text.empty())
            return;

        if (theField.Text.size() == 1 && theField.Text[0].empty())
            return;

        xml.writeElement("RT", theField.Text);
    }

    METHOD onSubmission(long theLineNumber, SubmissionInfo const & theField)
    {
        xml.writeBeginTag("SubmissionInfo");

        xml.writeElement("Submitted", theField.SubmissionDate);
        xml.writeElement("Info", theField.Info);

        xml.writeEndTag("SubmissionInfo");
    }

    METHOD onUnpublished(long theLineNumber, UnpublishedInfo const & theField)
    {
        xml.writeBeginTag("Unpublished");
        xml.writeEndTag("Unpublished");
    }

    METHOD onBook(long theLineNumber, BookInfo const & theField)
    {
        xml.writeBeginTag("Book");

        xml.writeElement("EditorList", "Editor", theField.Editors);
        xml.writeElement("Title", theField.Title);
        xml.writeElement("FirstPage", theField.FirstPage);
        xml.writeElement("LastPage", theField.LastPage);
        xml.writeElement("PublicationDate", theField.PublicationDate);
        xml.writeElement("Publisher", theField.Publisher);

        xml.writeEndTag("Book");
    }

    METHOD onThesis(long theLineNumber, ThesisInfo const & theField)
    {
        xml.writeBeginTag("Thesis");

        xml.writeElement("Year", theField.Year);
        xml.writeElement("Institute", theField.Institute);

        xml.writeEndTag("Thesis");
    }

    METHOD onPatent(long theLineNumber, PatentInfo const & theField)
    {
        xml.writeBeginTag("Patent");

        xml.writeElement("PatentNumber", theField.PatentNumber);
        xml.writeElement("Date", theField.Date);

        if (theField.Applicants.size() > 0)
            xml.writeElement("ApplicantList", "Applicant", theField.Applicants);

        xml.writeEndTag("Patent");
    }

    METHOD onOnlineJournal(long theLineNumber, OnlineJournalInfo const & theField)
    {
        xml.writeElement("OnlineJournal", "");
    }

    METHOD onElectronicResource(long theLineNumber, ElectronicResourceInfo const & theField)
    {
        xml.writeElement("ElectronicResource", theField.FreeText);
    }

	METHOD onMiscResource(long theLineNumber, MiscResourceInfo const & theField)
    {
        xml.writeElement("MiscResource", theField.FreeText);
    }

    METHOD onJournalArticle(long theLineNumber, JournalArticleInfo     const & theField)
    {
        xml.writeBeginTag("JournalArticle");

        xml.writeElement("Journal", theField.JournalName);
        xml.writeElement("Volume", theField.Volume);

        if (!theField.Issue.empty())
            xml.writeElement("Issue", theField.Issue);

        xml.writeElement("FirstPage", theField.FirstPage);
        xml.writeElement("LastPage", theField.LastPage);
        xml.writeElement("PublicationDate", theField.PublicationDate);

        xml.writeEndTag("JournalArticle");
    }

    METHOD onEndPublication()
    {
        xml.writeEndTag("Publication");
    }

private:
    XmlWriter xml;
};

int main(int argc, char* argv[])
{
    using namespace OpenEMBL;

    if (argc < 2)
        return 1;

    PARSER theParser(argv[1]);
    theParser.parse();

    return 0;
}

