#! /usr/bin/perl -w

# Released under the terms of the BiO Licence.
# http://biomatics.kaist.ac.kr/Research/Biolicense/

use strict;
use Getopt::Long;

use lib 		'/BiO/Research/Perl/Modules/';

use Mysql::Simple 	'doQuery';
use PSIMAP 		'$scopDb';

my $PDB;

GetOptions (
  "pdb=s"	=> \$PDB,
);

die( &usage ) unless $PDB;


my @color = qw(
  Red 
  Green 
  Blue 
  Cyan 
  Magenta 
  Yellow 
  Orange 
  SeaGreen 
  Pink 
  SkyBlue 
  Gold 
  Brown 
  HotPink 
  Purple 
  BlueTint 
  Grey
  GreenBlue 
  PinkTint 
  GreenTint  
  RedOrange 
  Violet 
  YellowTint
);


# Get domain data.

my $domain = 
  doQuery("
    SELECT PDB, SUNID, SID, SCCS, DOMAIN
    FROM $scopDb.domain
    WHERE PDB = '$PDB'
  ") or die "DBI::errstr\n";

die "$PDB NOT FOUND IN $scopDb\n" unless @$domain;


my $time = localtime();

print "
  load inline
  echo Protein Contact DB: $PDB\t$time
  echo Author:             Dan Bolser (MRC Dunn)
  echo \n\n";

my %sccs;	# Checks for
my %color;	# multi groups.

# For each domain

for (my $i=0; $i<@$domain; $i++){
  
  my ($pdb, $sunid, $sid, $sccs, $domainDef) = @{$domain->[$i]};
  
  # Define domain name alias
  my $color = $color[$i%@color];
  
  # Spit some information
  warn join("\t", $i, @{$domain->[$i]}, $color), "\n";
  
  # Format SCCS for rasmol
  my $sccs_name = $sccs;
     $sccs_name =~ tr/\./_/;
  
  
  
  # Deal with SCCS groups.
  unless ( $sccs{$sccs} ){
    print "
      define $sccs_name
    ";
    $sccs{$sccs}++;
  }
  
  # Deal with color groups.
  unless ( $color{$color} ){
    print "
      define x$color
    ";
    $color{$color}++;
  }
  
  
  
  # Format some text
  my $details = sprintf( "%12s: %3s %5s %5s %-15s %-15s %8s\n", 
    ("DOMAIN($i)", $pdb, $sunid, $sid, $sccs_name, $domainDef, $color));
  
  # Convert SCOP domain definition into rasmol format.
  my $selectDomain 
     = &scop2rasmol( $domainDef );
  
  
  # Do the main
  
  print "
    echo $details
    
    select $selectDomain				# Select domain
    color  $color\n					# Color domain.
    
    define px$sunid		 selected		# Define alias...
    define No$i			 selected		# ...
    
#    select within(5.0,selected)				# Select domain contacts!
    
#    define surf_px$sunid	 selected		# Define alias...
#    define surf_No$i		 selected
    
    
    select px$sunid					# Handle multi domain alias
    
    select x$color					# ...
    define x$color      selected, $selectDomain		# ...
        
    select $sccs_name					# ...
    define $sccs_name   selected, $selectDomain		# ...
    
    
    select !*						# Reset selection.
    \n\n";
}

print "
  select *
  \n
  echo 
  echo Each domain is defined as... 
  echo px00000		(scop sunid)
  echo a_1_1_1_1	(scop sccs)
  echo xCOLOR	 	(COLOR = rasmol color, eg xRed)
  echo No1		(n = domain number, eg No1)
  echo 
  \n
  exit					# These newlines are important!
  \n\n";
  
warn "OK\n";




sub scop2rasmol {
  my $scopDomainDefinition = shift;
  my @rasmolSelect;
  
  # Split the domain definition into components.
  
  foreach ( split(/,/, $scopDomainDefinition) ){
   
    if    ( /^(.{1}):$/o                                      ){ push @rasmolSelect,     "*$1" }	# X:
    elsif (        /^((?:-|)\d+)(?:\D|)-((?:-|)\d+)(?:\D|)$/o ){ push @rasmolSelect, "$1-$2"   }	#   40-90
    elsif ( /^(.{1}):((?:-|)\d+)(?:\D|)-((?:-|)\d+)(?:\D|)$/o ){ push @rasmolSelect, "$2-$3$1" }	# X:40-90
    elsif ( /^-$/o                                            ){ push @rasmolSelect, "*"       }	# -
    else  {
      die "$_:BAD DOMAIN DEF!\n"
    }
  }
  my $rasmolSelectString = join(",", @rasmolSelect);
  
  return $rasmolSelectString;
}

sub usage{
  warn <<"EOS";
  
  Create a rasmol script for defining the SCOP domain structure of a PDB. 

USAGE:
  $0 PDBCode

EOS
}

