package similarityValue;

import java.util.Iterator;
import java.util.regex.*;
import java.util.LinkedList;

import java.io.File;
import java.io.FileReader;
import java.io.BufferedReader;
import java.io.FileNotFoundException;
import java.io.IOException;

//Requires http://java.sun.com/products/java-media/3D/download.html
import javax.vecmath.Point3d;

public class Molecule implements Iterable<PotentialPoint> {
	LinkedList<PotentialPoint> potentials = new LinkedList<PotentialPoint>();
	long numPoints = 0;
	
	public Molecule (File potentials) throws FileNotFoundException, IOException {
		loadFromFile(potentials);

		//Make sure we loaded data
		assert(potentials.length() > 0);
	}
		
	public Iterator<PotentialPoint> iterator() {
		return potentials.iterator();
	}
	
	private void loadFromFile(File potentialsFile) throws FileNotFoundException, IOException {
		System.out.println("Loading " + potentialsFile.getAbsolutePath());
		
		
		String curLine;
		
		//Note: regex patterns must match the entire string
		
		String OneFloat = new String("([0-9.e+-]+)");
		String Point = ("\\((" + OneFloat + "), (" + OneFloat + "), (" + OneFloat + ")\\)");
		
		Pattern blankLine  =  Pattern.compile("^\\s*$"); //Unescaped: ^\s*$
		
		Pattern commentRegex = Pattern.compile("^(.*)#.*"); //Unescaped: ^\s*#.*
		
		Pattern numPointsRegex = Pattern.compile("^Number of points: " + OneFloat); //Unescaped:
		boolean numPointsFound = false;
		
		String PointsBeginLine = "=== Points ===";
		boolean foundPointsBeginLine = false;
		
		Pattern PointLine = Pattern.compile(Point + " " + OneFloat);
		
		BufferedReader inFile = new BufferedReader(new FileReader(potentialsFile));
		while ((curLine = inFile.readLine()) != null) {
			//Look for comments (Anything after # symbol)
			Matcher commentMatcher = commentRegex.matcher(curLine);
			if (commentMatcher.matches()) {
				// Remove the comment
				curLine = commentMatcher.group(1);
			}
			
			if (curLine.length() == 0 || blankLine.matcher(curLine).matches()) {
				//skip blank lines; goto next line
		        continue;
			}
			
			if (!numPointsFound) {
				Matcher m = numPointsRegex.matcher(curLine);
				if (m.matches()) {
					numPoints = Long.parseLong(m.group(1));
					numPointsFound = true;
				}
			}
			
			if (!foundPointsBeginLine) {
				foundPointsBeginLine = curLine.equals(PointsBeginLine);
			}
			else {
				//We're reading data
				Matcher m = PointLine.matcher(curLine);
				if (m.matches()) {
					potentials.push(new PotentialPoint(new Point3d(Double.parseDouble(m.group(1)),
															Double.parseDouble(m.group(2)),
															Double.parseDouble(m.group(3))),
														Double.parseDouble(m.group(4))));
					
					if(potentials.size() % 50000 == 0) {
						System.out.println((100.0 * potentials.size() / numPoints) + "% (" + potentials.size() + " of " + numPoints + ")");
					}
				}
			}
		}

		inFile.close();
	}
	
	public long getNumPoints() {
		return numPoints;
	}
}
