package pdb_reader;

import pdb_reader.data.spacegroup.SpaceGroup;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;
import java.util.TreeMap;
import java.util.TreeSet;
import javax.swing.event.EventListenerList;
import pdb_reader.data.*;

import pdb_editor.coordinate.CoordinateTable;


public class DataSet {
        static public TreeMap<String, String> ResidueTypeDatabase = new TreeMap<String, String> ();
        static public TreeMap<String, TreeMap<String, String>> ResidueAtomElementDatabase = new TreeMap<String, TreeMap<String, String>> ();
        static public TreeMap<String, Character> ResidueSingleLetterCodeDatabase = new TreeMap<String, Character> ();
        
        private AtomDataChangeEventListener ADCEL = new AtomDataChangeEventListener() {
            public void dataChanged(AtomDataChangeEvent e) {
                fireDataChangeEvent(e);
            }
        };
        
        private class ArrayListWithEvent<T> extends ArrayList<T> 
        {
            public boolean add(T value)
            {
                ((Atom)value).addAtomDataChangeListener(ADCEL);
                boolean rv = super.add(value);
                fireDataAddEvent((Atom)value);
                return rv;
            }
            
            public void add(int index, T value)
            {
                ((Atom)value).addAtomDataChangeListener(ADCEL);
                super.add(index, value);
                fireDataAddEvent((Atom)value, index);
            }
            
            public T remove(int index)
            {
                fireDataDeleteEvent((Atom)this.get(index), index);
                return super.remove(index);
            }
            
            public boolean remove(Object o)
            {
                fireDataDeleteEvent((Atom)o, this.indexOf(o));
                return super.remove(o);
            }
        }
        
        private CoordinateTable tablelinked = null;
        
	private String fileName = null;
	private String[] originalData = null;
	private ArrayListWithEvent<Atom> atoms = null;
	private SpaceGroup spacegroup = null;
        private EventListenerList dataChangeListener = new EventListenerList();
	
        private TreeMap<Atom, Connectivity> connectivity = null;
        
	// Constructors
	public DataSet()
	{
		fileName = null;
		atoms = new ArrayListWithEvent<Atom> ();
	}
	public DataSet(String Filename) 
	{ 
		ReadPDBFile(Filename);
	}
	
        
	public CoordinateTable TableLinked() { return tablelinked; }
        public void TableLinked(CoordinateTable t) { tablelinked = t; }
	
	// Interface functions
	//   Output functions
	public String FileName() { return fileName; }
	public ArrayList<Atom> Atoms() { return atoms; }
        public SpaceGroup Spacegroup() { return spacegroup; }
        public TreeMap<Atom, Connectivity> ConnectList() { return connectivity; }        
        //   Input functions
	public String FileName(String Value) { return fileName = Value; }
	public SpaceGroup Spacegroup(SpaceGroup Value) { 
            for (int i=0; i<atoms.size(); i++) atoms.get(i).Spacegroup(Value);
            return spacegroup = Value; }
	public void Atoms(ArrayList<Atom> Value) { atoms = (ArrayListWithEvent<Atom>)Value; }
        public TreeMap<Atom, Connectivity> ConnectList(TreeMap<Atom, Connectivity> Value) { return connectivity = Value; }        
	
	// Public functions
        public void UpdateConnectivityTable()
        {
            TreeSet<Connectivity> removelist = new TreeSet<Connectivity> ();
            
            for (Connectivity c : connectivity.values())
            {
                if (!atoms.contains(c.Base)) { removelist.add(c); continue; }
                Set<Atom> list = c.Branch;
                for (Atom a : list) if (!atoms.contains(a)) list.remove(a);
            }
            
            for (Connectivity c : removelist)
                connectivity.remove(c.Base);
            
            /*for (int i=connectivity.size() - 1; i >=0; i++)
            {
                Connectivity c = connectivity.get(i);
                if (!atoms.contains(c)) { connectivity.remove(i); continue; }
                ArrayList<Atom> removelist = new ArrayList<Atom> ();
                Set<Atom> list = c.Branch;
                for (Atom a : list) if (!atoms.contains(a)) removelist.add(a);
                if (list.size() > 0) list.removeAll(list);
            }*/
        }
        
        
        public void InsertEmptyAtom(int row)
        {
            Atom a = new Atom(spacegroup);
            atoms.add(row, a);
        }
        
        public void ReduceProteinResiduesToCa()
        {
            for (int i=atoms.size()-1; i>=0; i--)
            {
                Atom a = atoms.get(i);
                if (ResidueTypeDatabase.containsKey(a.ResidueType()))
                    if (ResidueTypeDatabase.get(a.ResidueType()).equals("Protein"))
                        if (!a.AtomType().equals("CA"))
                            atoms.remove(i);
            }
        }
        
        public void ReduceProteinResiduesToCa(int[] selectedIndicies)
        {
            for (int i=selectedIndicies.length-1; i>=0; i--)
            {
                Atom a = atoms.get(selectedIndicies[i]);
                if (ResidueTypeDatabase.containsKey(a.ResidueType()))
                    if (ResidueTypeDatabase.get(a.ResidueType()).equals("Protein"))
                        if (!a.AtomType().equals("CA"))
                            atoms.remove(i);
            }
        }
        
        
        public void ReduceProteinResiduesToPolyALA()
        {
            String[] Keep = {"N", "C", "O", "CA", "HA", "H", "H1", "H2", "H3", "CB", "HB", "HB1", "HB2", "HB3"};
            String[] AllowedRes = {"ALA", "GLY"};
            
            TreeSet<String> KeepAtoms = new TreeSet<String>();
            for (int i=0; i<Keep.length; i++) KeepAtoms.add(Keep[i]);
            
            for (int i=atoms.size()-1; i>=0; i--)
            {
                Atom a = atoms.get(i);
                if (ResidueTypeDatabase.containsKey(a.ResidueType()))
                    if (ResidueTypeDatabase.get(a.ResidueType()).equals("Protein"))
                    {
                        if (!KeepAtoms.contains(a.AtomType())) { atoms.remove(i); continue; }
                        if (!Global.StringArrayContains(AllowedRes, a.ResidueType()))
                        { a.ResidueType("ALA"); continue;}
                    }
            }
        }
        
        public void ReduceProteinResiduesToPolyALA(int[] selectedIndicies)
        {
            String[] Keep = {"N", "C", "O", "CA", "HA", "H", "H1", "H2", "H3", "CB", "HB", "HB1", "HB2", "HB3"};
            String[] AllowedRes = {"ALA", "GLY"};
            
            TreeSet<String> KeepAtoms = new TreeSet<String>();
            for (int i=0; i<Keep.length; i++) KeepAtoms.add(Keep[i]);
            
            for (int i=selectedIndicies.length-1; i>=0; i--)
            {
                Atom a = atoms.get(selectedIndicies[i]);
                if (ResidueTypeDatabase.containsKey(a.ResidueType()))
                    if (ResidueTypeDatabase.get(a.ResidueType()).equals("Protein"))
                    {
                        if (!KeepAtoms.contains(a.AtomType())) { atoms.remove(i); continue; }
                        if (!Global.StringArrayContains(AllowedRes, a.ResidueType()))
                        { a.ResidueType("ALA"); continue;}
                    }
            }
        }
        
        public void Sort(final int[] dataIndicies, final boolean[] descending)
        {
            this.fireDataChangeEvent(new DataSetDataChangeEvent(this,DataSetDataChangeEvent.SORT_EVENT));
            Collections.sort(atoms, new Comparator<Atom> () {
                public int compare(Atom o1, Atom o2) {
                    for (int i=0; i<dataIndicies.length; i++)
                    {
                        if (dataIndicies[i] < Atom.TOTAL_NUMBER_OF_INDEX)
                        {
                            int compvalue = Atom.compareAscending(o1, o2, dataIndicies[i]);
                            if (compvalue != 0) 
                            {
                                if (descending[i]) compvalue *= -1;
                                return compvalue;
                            }
                        }
                    }
                    return 0;
                }
            });            
        }
        
        public void Sort(final int dataIndex, final boolean descending)
        {
            this.fireDataChangeEvent(new DataSetDataChangeEvent(this,DataSetDataChangeEvent.SORT_EVENT));
            Collections.sort(atoms, new Comparator<Atom> () {
                public int compare(Atom o1, Atom o2) {
                    if (descending) return Atom.compareAscending(o1, o2, dataIndex) * -1;
                    return Atom.compareAscending(o1, o2, dataIndex);
                }
                
            });
        }
        
        public int[] MoveTopIndex(int[] selectedIndicies)
        {
            for (int i=0; i<selectedIndicies.length; i++)
            {
                int index = selectedIndicies[i];
                if (index > i)
                {
                    Atom a = atoms.remove(index);
                    atoms.add(i, a);
                    selectedIndicies[i] = i;
                }
            }
            return selectedIndicies;            
        }
        
        public int[] MoveBottomIndex(int[] selectedIndicies)
        {
            int max = atoms.size() - 1;
            for (int i=selectedIndicies.length - 1; i>=0; i--)
            {
                int index = selectedIndicies[i];
                if (index < max)
                {
                    Atom a = atoms.remove(index);
                    atoms.add(max, a);
                    selectedIndicies[i] = max;
                }
                max--;
            }
            return selectedIndicies;
        }
        
        public int[] MoveUpIndex(int[] selectedIndicies)
        {
            for (int i=0; i<selectedIndicies.length; i++)
            {
                int index = selectedIndicies[i];
                if (index > i)
                {
                    Atom a = atoms.remove(index);
                    atoms.add(--index, a);
                    selectedIndicies[i] = index;
                }
            }
            return selectedIndicies;
        }
        
        public int[] MoveDownIndex(int[] selectedIndicies)
        {
            int max = atoms.size() - 1;
            for (int i=selectedIndicies.length - 1; i>=0; i--)
            {
                int index = selectedIndicies[i];
                if (index < max--)
                {
                    Atom a = atoms.remove(index);
                    atoms.add(++index, a);
                    selectedIndicies[i] = index;
                }
            }
            return selectedIndicies;
        }
        
        public Atom[] CopyAtoms(boolean AddToDatabase, int[] rows)
        {
            Atom[] selected = this.getAtoms(rows);
            Atom[] r = new Atom[rows.length];
            int atomnumber = atoms.get(atoms.size()-1).AtomNumber();
            for (int i=0; i<r.length; i++)
            {
                r[i] = selected[i].clone();
                r[i].AtomNumber(++atomnumber);
                if (AddToDatabase) atoms.add(r[i]);
            }
           
            return r;
        }
        
        public Atom[] CopyAtoms(boolean AddToDatabase, int[] rows, int[] NewValueIndicies, Object[] NewValues)
        {
            Atom[] selected = this.getAtoms(rows);
            Atom[] r = new Atom[rows.length];
            int atomnumber = atoms.get(atoms.size()-1).AtomNumber();
            for (int i=0; i<r.length; i++)
            {
                r[i] = selected[i].clone();
                r[i].AtomNumber(++atomnumber);
                for (int j=0; j<NewValueIndicies.length; j++)
                    r[i].TableData(NewValueIndicies[j], NewValues[j]);
                if (AddToDatabase) atoms.add(r[i]);
            }
           
            return r;
        }
        
        public void UpdateScaleMatrixChange()
        {
            for (int i=0; i<atoms.size(); i++)
                atoms.get(i).OrthogonalCoordinate(atoms.get(i).OrthogonalCoordinate());
        }
        
        public void TransformAtoms(int[] indicies, double[][] matrix4x4, char newchainid)
        {
            Atom[] selected = this.getAtoms(indicies);
            Atom[] copied = new Atom[selected.length];
            int atomnumber = atoms.get(atoms.size()-1).AtomNumber();
            for (int i=0; i<copied.length; i++)
            {
                copied[i] = selected[i].clone();
                copied[i].ChainID(newchainid);
                copied[i].AtomNumber(++atomnumber);
                atoms.add(copied[i]);
            }
            TransformAtoms(copied, matrix4x4);
        }
        
        public void TransformAtoms(int[] indicies, double[][] matrix4x4)
        {
            TransformAtoms(this.getAtoms(indicies), matrix4x4);
        }
        
        public void TransformAtoms(Atom[] atoms, double[][] matrix4x4)
        {
            for (int i=0; i<atoms.length; i++)
                atoms[i].OrthogonalCoordinate(Global.TransformCoordinates(matrix4x4, atoms[i].OrthogonalCoordinate()));
        }
        
        public TreeMap<String, double[]> calculateStatistics()
        {
            return calculateStatistics(atoms.toArray(new Atom[atoms.size()]));
        }

        public TreeMap<String, double[]> calculateStatistics(int[] rows)
        {
            Atom[] selected = new Atom[rows.length];
            for (int i=0; i<selected.length; i++)
                selected[i] = atoms.get(rows[i]);
            return calculateStatistics(selected);
        }
        
        public TreeMap<String, double[]> calculateStatistics(Atom[] list)
        {
            TreeMap<String, double[]> r = new TreeMap<String, double[]> ();
            
            for (int col = 0; col<Atom.TOTAL_NUMBER_OF_INDEX; col++)
            {
                if ((Atom.DataClass[col].equals(Global.integerclass.getClass())) |
                        (Atom.DataClass[col].equals(Global.doubleclass.getClass())))
                {
                ArrayList<Double> values = new ArrayList<Double> ();
                for (int i=0; i<list.length; i++)
                {
                    if (list[i].TableData(col) != null)
                    {
                        if (list[i].TableData(col).getClass().equals(Global.doubleclass.getClass()))
                            values.add((Double)list[i].TableData(col));
                        if (list[i].TableData(col).getClass().equals(Global.integerclass.getClass()))
                            values.add((double)((Integer)list[i].TableData(col)));
                    }
                }
                if (values.size() > 0) 
                {
                    double[] calcstat = Global.calculateAvgSDMaxMin(values.toArray(new Double[values.size()]));
                    int maxindex = (int)calcstat[Global.calculateAvgSDMaxMin_MaxRowIndex];
                    calcstat[Global.calculateAvgSDMaxMin_MaxRowIndex] = list[maxindex].AtomNumber();
                    int minindex = (int)calcstat[Global.calculateAvgSDMaxMin_MinRowIndex];
                    calcstat[Global.calculateAvgSDMaxMin_MinRowIndex] = list[minindex].AtomNumber();
                    r.put(Atom.DataToolTip[col], calcstat);
                }
                }
            }

            return r;
        }
        
        public Integer[] getIndicesOfResidueType(String Type)
        {
            return getIndicesOfResidueType(Type, 0, atoms.size() - 1);
        }
        
        public Integer[] getIndicesOfResidueType(String Type, int startindex, int endindex)
        {
             ArrayList<Integer> r = new ArrayList<Integer> ();
             
             for (int i=startindex; i<=endindex; i++)
             {
                 String residue = atoms.get(i).ResidueType();
                 if (ResidueTypeDatabase.containsKey(residue))
                     if (ResidueTypeDatabase.get(residue).equals(Type))
                         r.add(i);
             }
             
             return r.toArray(new Integer[r.size()]);
        }
        
        public Integer[] getIndicesOfUnclassifiedType()
        {
            return getIndicesOfUnclassifiedType(0, atoms.size() - 1);
        }
        
        public Integer[] getIndicesOfUnclassifiedType(int startindex, int endindex)
        {
             ArrayList<Integer> r = new ArrayList<Integer> ();
             
             for (int i=startindex; i<=endindex; i++)
             {
                 String residue = atoms.get(i).ResidueType();
                 if (!ResidueTypeDatabase.containsKey(residue))
                     r.add(i);
             }
             
             return r.toArray(new Integer[r.size()]);
        }
        
        public Integer[] fixElementInformation()
        {
            ArrayList<Integer> changedindicies = new ArrayList<Integer> ();
            for (int i=0; i<atoms.size(); i++)
            {
                Atom a = atoms.get(i);
                String r = a.ResidueType();
                String t = a.AtomType();
                String e = a.Element();
                
                if ((ResidueAtomElementDatabase.containsKey(r)) && (ResidueAtomElementDatabase.get(r).containsKey(t)))
                {
                    boolean fix = false;
                    
                    if (e != null)
                    { if (!e.equals(ResidueAtomElementDatabase.get(r).get(t))) fix = true; }
                    else fix = true;
                        
                    
                    if (fix)
                    {
                        a.Element(ResidueAtomElementDatabase.get(r).get(t));
                        changedindicies.add(i);
                    }
                }
                else if (ResidueTypeDatabase.containsKey(r))
                {
                    String type = ResidueTypeDatabase.get(r);
                    if ((type.equals("Protein")) || (type.equals("DNA")) || (type.equals("RNA")))
                    {
                        if (Character.isLetter(t.charAt(0))) a.Element(""+t.charAt(0));
                        else a.Element(""+t.charAt(1));
                        changedindicies.add(i);
                    }
                }
            }
            if (changedindicies.size() > 0)
                return changedindicies.toArray(new Integer[changedindicies.size()]);
            return null;
        }
        
        public int deleteAlternateConformation()
        {
                int count = 0;
                for (int j=atoms.size() - 1; j>=0; j--)
                {
                    Atom b = atoms.get(j);
                    if (b.AlternateLocation() == 'A')
                    {
                        b.AlternateLocation(' ');
                        b.Occupancy(1);
                    }
                    if ((b.AlternateLocation() != ' ') & (b.AlternateLocation() != 'A'))
                    {
                        atoms.remove(j);
                        count++;
                    }    
                }
                return count;
        }
        
        public int deleteAlternateConformation(int[] indicies)
        {
            int count = 0;
            Atom[] selected = new Atom[indicies.length];
            for (int i=0; i<indicies.length; i++)
                selected[i] = atoms.get(indicies[i]);
            for (int i=0; i<selected.length; i++)
            {
                Atom b = selected[i];
                if (b.AlternateLocation() == 'A')
                {
                    b.AlternateLocation(' ');
                    b.Occupancy(1);
                }
                if ((b.AlternateLocation() != ' ') & (b.AlternateLocation() != 'A'))
                {
                    atoms.remove(b);
                    count++;
                }    
            }
            return count;
        }
        
        public Atom[] getAtoms(int[] indicies)
        {
            Atom[] r = new Atom[indicies.length];
            for (int i=0; i<indicies.length; i++)
                r[i] = atoms.get(indicies[i]);
            return r;
        }
        
        public int[] getChainIndicies(int[] indicies)
        {
            HashSet<Character> list = new HashSet<Character> ();
            for (int i=0; i<indicies.length; i++)
                list.add(atoms.get(indicies[i]).ChainID());
            
            ArrayList<Integer> r = new ArrayList<Integer> ();
            for (int i=0; i<atoms.size(); i++)
                if (list.contains(atoms.get(i).ChainID()))
                    r.add(i);
            
            int[] result = new int[r.size()];
            for(int i=0; i<result.length; i++)
                result[i] = r.get(i);
            return result;
        }
        
        public int[] getResiduesIndicies(int indicies[])
        {
            TreeSet<Integer> found = new TreeSet<Integer> ();
            for (int i=0; i<indicies.length; i++)
            {
                Integer[] residue = getResidueIndicies(atoms.get(indicies[i]));
                for (int j=0; j<residue.length; j++)
                    found.add(residue[j]);
            }
            return Global.ConvertIntegerArrayToIntArray(found.toArray(new Integer[found.size()]));
        }
        
        public Atom[] getResidue(int index)
        {
            return getResidue(atoms.get(index));
        }
        
        public Atom[] getResidue(Atom a)
        {
            ArrayList<Atom> found = new ArrayList<Atom> ();

            for (int i=0; i<atoms.size(); i++)
            {
                Atom b = atoms.get(i);
                if (CheckSameResidue(a, b)) found.add(b);
            }
            
            return found.toArray(new Atom[found.size()]);
        }
        
        public Integer[] getResidueIndicies(Atom a)
        {
            ArrayList<Integer> found = new ArrayList<Integer> ();

            for (int i=0; i<atoms.size(); i++)
            {
                Atom b = atoms.get(i);
                if (CheckSameResidue(a, b)) found.add(i);
            }
            
            return found.toArray(new Integer[found.size()]);
        }
        
        public int deleteSelectorAtoms(AtomSelector selector)
        {
            return deleteSelectorAtoms(selector, 0, atoms.size()-1);
        }
        public int deleteSelectorAtoms(AtomSelector selector, int startindex, int endindex)
        {
            int[] indicies = new int[endindex-startindex+1]; 
            int j = 0;
            for (int i=startindex; i<=endindex; i++)
                indicies[j++] = i;
            return deleteSelectorAtoms(selector, indicies);
        }
        public int deleteSelectorAtoms(AtomSelector selector, int[] indicies)
        {
            int count = 0;
            for (int i=indicies.length - 1; i>=0; i--)
            {
                if (selector.selectAtom(atoms.get(indicies[i])))
                {
                    count++;
                    atoms.remove(indicies[i]);
                }
            }
            return count;
        }
        
        public String getReadingError()
        {
            StringBuilder sb = new StringBuilder();
            
            String coordinateErrors = GenerateAtomReadingErrorMessage();
            if (!coordinateErrors.equals(""))
            {
                sb.append("Could not process following coordinate records :\n");
                sb.append(coordinateErrors);
            }
            return sb.toString();
        }
        
        public int findIndex(int AtomNumber)
        {
            for (int i=0; i<atoms.size(); i++)
                if (atoms.get(i).AtomNumber() == AtomNumber)
                    return i;
            return -1;
        }
        
        public String SaveFile(String FileName)
        {
		fileName = FileName;
		return SaveFile();
	}
	
	public String SaveFile()
	{
            try {
                Global.SaveTextFile(fileName, GetFullPDBText());
                return null;
            }
            catch (Exception e)
            {
                return e.getMessage();
            }
	}
	
	public void ReadPDBFile(String Filename)
	{
		fileName = Filename;
		ReadPDBFile();
	}
	
	public void DeleteAtoms(int[] records)
	{
		for (int i=records.length-1; i>=0; i--)
		{
			atoms.remove(records[i]);
		}
	}

        public void ResetAtomNumbers()
        {
            int i = 1;
            Iterator e = atoms.iterator();
            while(e.hasNext())
                ((Atom)e.next()).AtomNumber(i++);
        }
	
        public int AppendPDBInformation(String s, int insertindex)
        {
            ArrayList<Atom> l = this.ExtractAtomInformation(s);
            if (l != null)
            {
                if (l.size() > 0)
                {
                    int j =0;
                    Iterator i = l.iterator();
                    while (i.hasNext()) atoms.add(insertindex+j++,(Atom)i.next());
                    UpdateAssociatedTable();
                    return l.size();
                }
            }
            return 0;
        }
                
	
	// Private functions
        private boolean CheckSameResidue(Atom a, Atom b)
        {
            Object[] criteria = new Object[] {a.ChainID(), a.ResidueNumber(), a.ResidueType()};
            Object[] search = new Object[] {b.ChainID(), b.ResidueNumber(), b.ResidueType()};
            for (int j=0; j<criteria.length; j++)
            {
                if (!criteria[j].equals(search[j]))
                    return false;
            }
            return true;
        }
        
        private void UpdateAssociatedTable()
        {
            if (tablelinked != null) tablelinked.fireTableDataChanged();
        }
        
	private String GetFullPDBText()
	{
		StringBuilder sb = new StringBuilder();
		
		boolean DidCoordinates = false;
		boolean DidSpaceGroup = false;
                boolean DidConnectivity = false;
                
                if (originalData != null) {
		for (int i=0; i<originalData.length; i++)
		{
                        if (originalData[i] != null)
			if (originalData[i].length() > 6)
                        {
                            if (CheckPDBSpaceGroupLine(originalData[i]))
                            {
                                if (!DidSpaceGroup)
                                {
                                    sb.append(spacegroup.WritePDBSpaceGroupLines());
                                    DidSpaceGroup = true;
                                }
                                continue;
                            }
                 	    if (CheckPDBAtomLine(originalData[i]) || CheckPDBAtomAppendingLine(originalData[i])
                                    || CheckPDBAtomIgnoreLine(originalData[i]))
                            {
                                if (DidCoordinates == false)
				{
                                    if (!DidSpaceGroup)
                                    {
                                        sb.append(spacegroup.WritePDBSpaceGroupLines());
                                        DidSpaceGroup = true;
                                    }
                                    sb.append(GetPDBCoordinates());
                                    DidCoordinates = true;
				}
				continue;
                            }
                            if (CheckPDBConnectivityLine(originalData[i]))
                            {
                                if (!DidConnectivity)
                                {
                                    if (!DidSpaceGroup)
                                    {
                                        sb.append(spacegroup.WritePDBSpaceGroupLines());
                                        DidSpaceGroup = true;
                                    }
                                    if (!DidCoordinates)
                                    {
                                        sb.append(GetPDBCoordinates());
                                        DidCoordinates = true;
                                    }
                                    sb.append(WritePDBConnectivityLines());
                                    DidConnectivity = true;
                                }
                                continue;
                            }
                        }
			if (!CheckPDBSkipLine(originalData[i])) sb.append(originalData[i] + '\n');
		}}
                
                if (!DidSpaceGroup) sb.append(spacegroup.WritePDBSpaceGroupLines());
                if (!DidCoordinates) sb.append(GetPDBCoordinates());
                if (!DidConnectivity) sb.append(WritePDBConnectivityLines());
                
                sb.append("END");
                
		return sb.toString();
	}
	
       
        private boolean CheckPDBSkipLine(String line)
        {
            String[] list = {"MASTER", "END"};
            if (line != null)
            {
                for (String s : list)
                {
                    if (line.length() > s.length())
                        if (line.substring(0, s.length()).equals(s)) return true;
                }
            }
            return false;
        }
        
        private boolean CheckPDBConnectivityLine(String line)
        {
            if (line.substring(0, Connectivity.PDBHeader.length()).equals(Connectivity.PDBHeader)) return true;
            return false;
        }
        
        private String WritePDBConnectivityLines()
        {
            StringBuilder sb = new StringBuilder();
            
            this.UpdateConnectivityTable();
            for (Connectivity c : connectivity.values())
            {
                sb.append(c.WritePDBLine());
                sb.append('\n');
            }
            
            return sb.toString();
        }
        
	private String GetPDBCoordinates()
	{
		StringBuilder sb = new StringBuilder();
		
                for (int i=0; i<atoms.size(); i++)
                {
                    AtomPDB a = atoms.get(i).GetPDBAtomOfThis();
                    sb.append(a.WritePDBLine());
                    if ((i+1 >= atoms.size()) || (atoms.get(i+1).ChainID() != a.ChainID()))
                        sb.append(a.WritePDBTerLine());
                }
		
		return sb.toString();
	}
	
	
	private void ReadPDBFile()
	{
		String[] lines = Global.ReadTextFile(fileName);
		originalData = lines;
		
                spacegroup = ExtractSpaceGroupInformation(lines);
		atoms = ExtractAtomInformation(lines);
                connectivity = ExtractConnectivity(lines);
	}
        
        private TreeMap<Atom, Connectivity> ExtractConnectivity(String[] lines)
        {
            TreeMap<Atom, Connectivity> r = new TreeMap<Atom, Connectivity>();
            
            HashSet<Atom> base = new HashSet<Atom> ();
            for (String s : lines)
            {
                if (s != null)
                    if (s.length() > 6)
                        if (s.substring(0, 6).equals(Connectivity.PDBHeader))
                        {
                            Connectivity c = new Connectivity(s, atoms);
                            if (!r.containsKey(c.Base))
                                r.put(c.Base, c);
                            else
                                r.get(c.Base).Branch.addAll(c.Branch);
                        }
            }
            return r;
        }
        
        private SpaceGroup ExtractSpaceGroupInformation(String[] lines)
        {
            SpaceGroup result = new SpaceGroup();
            result.ReadFromPDBLines(lines);
            return result;
        }
        
	private ArrayListWithEvent<Atom> ExtractAtomInformation(String text)
        {
            String[] lines = text.split("\\n");
            return ExtractAtomInformation(lines);
        }
        
        private ArrayListWithEvent<Atom> ExtractAtomInformation(String[] lines)
	{
		ArrayListWithEvent<Atom> result = new ArrayListWithEvent<Atom> ();
		for (int i=0; i < lines.length; i++)
		{
			if (lines[i].length() >= 6)
			if (CheckPDBAtomLine(lines[i]))
			{
				AtomPDB a = new AtomPDB(lines[i], spacegroup);
				while ((i+1) < lines.length)
				{
					if (lines[i+1].length() >= 6)
					{
						if (CheckPDBAtomAppendingLine(lines[i + 1]))
							a.AppendPDBInformation(lines[++i]);
						else break;
					}
					else break;
				}
				result.add(a);
			}
		}
                return result;
	}
	
        private boolean CheckPDBSpaceGroupLine(String line)
        {
            return Global.StringArrayContains(SpaceGroup.Headers, line.substring(0, 5));
        }
        
        private boolean CheckPDBAtomLine(String line)
	{
		return Global.StringArrayContains(AtomPDB.Headers, line.substring(0, 6));
	}
	private boolean CheckPDBAtomAppendingLine(String line)
	{
		return Global.StringArrayContains(AtomPDB.ContinueHeaders, line.substring(0, 6));
	}
        private boolean CheckPDBAtomIgnoreLine(String line)
        {
		return Global.StringArrayContains(AtomPDB.IgnoreHeaders, line.substring(0, 6));
        }
        
        private String GenerateAtomReadingErrorMessage()
        {
            StringBuilder sb = new StringBuilder();
            
            for (int i=0; i<atoms.size(); i++)
            {
                Atom a = atoms.get(i);
                boolean[] errors = a.getReadingErrorRecords();
                if (errors[Atom.TOTAL_NUMBER_OF_INDEX])
                {
                    sb.append(a.AtomNumber());
                    sb.append(" : ");
                    for (int j=0; j<errors.length - 1; j++)
                        if (errors[j]) sb.append(Atom.DataToolTip[j] + ' ');
                    sb.append('\n');
                }
            }
            
            return sb.toString();
        }

        public void addDataChangeListener(DataSetDataChangeEventListener listener)
        {
            dataChangeListener.add(DataSetDataChangeEventListener.class, listener);
        }
        
        public void removeDataChangeListener(DataSetDataChangeEventListener listener)
        {
            dataChangeListener.remove(DataSetDataChangeEventListener.class, listener);
        }
        
        private void fireDataChangeEvent(AtomDataChangeEvent e)
        {
            DataSetDataChangeEvent event = new DataSetDataChangeEvent(this, e);
            fireDataChangeEvent(event);
        }

        private void fireDataAddEvent(Atom AtomAdded, int Index)
        {
            DataSetDataChangeEvent event = new DataSetDataChangeEvent(this, DataSetDataChangeEvent.ADD_EVENT, AtomAdded, Index);
            fireDataChangeEvent(event);
        }
        
        private void fireDataAddEvent(Atom AtomAdded)
        {
            DataSetDataChangeEvent event = new DataSetDataChangeEvent(this, DataSetDataChangeEvent.ADD_EVENT, AtomAdded);
            fireDataChangeEvent(event);
        }
        
        private void fireDataDeleteEvent(Atom AtomDeleted, int lastIndex)
        {
            DataSetDataChangeEvent event = new DataSetDataChangeEvent(this, DataSetDataChangeEvent.DELETE_EVENT, AtomDeleted, lastIndex);
            fireDataChangeEvent(event);
        }
        
        public void fireDataChangeEvent(DataSetDataChangeEvent event)
        {
            Object[] listeners = dataChangeListener.getListenerList();
            for (int i=0; i<listeners.length; i+=2) {
                if (listeners[i]==DataSetDataChangeEventListener.class) {
                    ((DataSetDataChangeEventListener)listeners[i+1]).dataChanged(event);
                }
            }
        }
        
        
        
        static public void initializeResidueTypeDatabase()
        {
            String[] s = Global.ReadTextResource("/pdb_editor/data/ResidueTypes.txt");
            for (int i=0; i<s.length; i++)
            {
                String[] entry = s[i].split("\t");
                if (entry.length == 2)
                {
                    ResidueTypeDatabase.put(entry[0], entry[1]);
                }
            }
        }
    
        
        static public void initializeResidueAtomElementDatabase()
        {
            String[] s = Global.ReadTextResource("/pdb_editor/data/ResidueAtomElementList.txt");
            for (int i=0; i<s.length; i++)
            {
                String[] entry = s[i].split("\t");
                if (entry.length == 3)
                {
                    if (!ResidueAtomElementDatabase.containsKey(entry[0])) ResidueAtomElementDatabase.put(entry[0], new TreeMap<String, String> ());
                    ResidueAtomElementDatabase.get(entry[0]).put(entry[1], entry[2]);
                }
            }
        }
        
        static public void initializeResidueSingleLetterCodeDatabase()
        {
            String[] s = Global.ReadTextResource("/pdb_editor/data/ResidueSingleLetterCodes.txt");
            for (int i=0; i<s.length; i++)
            {
                String[] entry = s[i].split("\t");
                ResidueSingleLetterCodeDatabase.put(entry[0], entry[1].charAt(0));
            }
        }
}

