package pdb_reader.data.spacegroup;

import java.util.ArrayList;
import java.util.TreeMap;
import pdb_reader.Global;

public class SpaceGroup {
    
    static final public int CRYST_INDEX = 0;
    static final public int ORIGX_INDEX = 1;
    static final public int SCALE_INDEX = 2;
    static final public String[] Headers = {"CRYST", "ORIGX", "SCALE"};
    
    static public TreeMap<String, SpaceGroupInformation> SpaceGroupTableByName = null;
    static public TreeMap<Integer, SpaceGroupInformation> SpaceGroupTableByNumber = null;
    static public void initSpaceGroupTable()
    {
        String[] lines = Global.ReadTextResource("/pdb_reader/data/spacegroup/SymmOp.lib");
        SpaceGroupTableByName = new TreeMap<String, SpaceGroupInformation> ();
        SpaceGroupTableByNumber = new TreeMap<Integer, SpaceGroupInformation> ();
        
        ArrayList<String> sgline = null;
        for (int i=0; i<lines.length; i++)
        {
            if (lines[i].charAt(0) != ' ')
            {
                if (sgline != null)
                {
                    SpaceGroupInformation sgi = new SpaceGroupInformation(sgline.toArray(new String[sgline.size()]));
                    if (!SpaceGroupTableByName.containsKey(sgi.Name)) SpaceGroupTableByName.put(sgi.Name, sgi);
                    if (!SpaceGroupTableByName.containsKey(sgi.Name2))SpaceGroupTableByName.put(sgi.Name2, sgi);
                    //SpaceGroupTableByName.put(sgi.Name3, sgi);
                    SpaceGroupTableByNumber.put(sgi.CCP4Number, sgi);
                }
                sgline = new ArrayList<String> ();
            }
            sgline.add(lines[i]);
        }
    }
        
 private double a = 0;
 private double b = 0;
 private double c = 0;
 private double alpha = 0;
 private double beta = 0;
 private double gamma = 0;
 private String name = null;
 private int number = 0;
 
 private String[][] symmetryOperators = null;
 
 private double[][] scale = null;
 private double[][] scaleInverse = null;
 
 private double[][] origx = null;
 private double[][] origxInverse = null;
 private boolean isOrigxIdentity = true;
 
 private double[] translationVector = null;
 private double[][][] NCSmatrix = null;

 private CrystalSystemChecker cellChecker = null;
 
 // Constructors
 public SpaceGroup() {}
 
 public SpaceGroup(String[] PDBLines)
 {
     this.ReadFromPDBLines(PDBLines);
 }
 
 public SpaceGroup(String PDBLines)
 {
     this.ReadFromPDBLines(PDBLines.split("\\n"));
 }
 
 // Interface functions
 //   Output functions
 public double a() { return a; }
 public double b() { return b; }
 public double c() { return c; }
 public double alpha() { return alpha; }
 public double beta() { return beta; }
 public double gamma() { return gamma; }
 public String name() { return name; }
 public int number() { return number; }
 public double[][] scale() { return scale; }
 public double[][] scaleInverse() { return scaleInverse; }
 public double[][] origx() { return origx; }
 public double[][] origxInverse() { return origxInverse; }
 public boolean isOrigxIdentity() {return isOrigxIdentity; }
 public double[] translationVector() { return translationVector; }
 public String[][] symmetryOperators() { return symmetryOperators; }
 public String crystalSystemText() { return cellChecker.InformationText(); }
 //   Input functions
 public double a(double Value) { return a = Value; }
 public double b(double Value) { return b = Value; }
 public double c(double Value) { return c = Value; }
 public double alpha(double Value) { return alpha = Value; }
 public double beta(double Value) { return beta = Value; }
 public double gamma(double Value) { return gamma = Value; }
 public String name(String Value) { GetSpaceGroupInformation(Value); return name; }
 public int number(int Value) { GetSpaceGroupInformation(Value); return number; }
 public double[][] scale(double[][] Value) { scale = Value; scaleInverse = Global.Inverse4x4Matrix(scale); return scale; }
 public double[][] scaleInverse(double[][] Value) { scaleInverse = Value; scale = Global.Inverse4x4Matrix(scaleInverse); return scaleInverse; }
 public double[][] origx(double[][] Value)
 { 
  origx = Value; 
  origxInverse = Global.Inverse4x4Matrix(origx); 
  isOrigxIdentity = Global.IsIdentityMatrix(origx);
  return origx; 
 }
 public double[][] origxInverse(double[][] Value)
 { 
  origxInverse = Value; 
  origx = Global.Inverse4x4Matrix(origxInverse);
  isOrigxIdentity = Global.IsIdentityMatrix(origx);
  return origxInverse; 
 }
 public double[] translationVector(double[] Value) { return translationVector = Value; }

 
 
 // Public functions
 public SpaceGroup clone()
 {
     SpaceGroup r = new SpaceGroup();

     r.a = this.a;
     r.b = this.b;
     r.c = this.c;
     
     r.alpha = this.alpha;
     r.beta = this.beta;
     r.gamma = this.gamma;
     
     r.name = this.name;
     r.number = this.number;
     
     if (symmetryOperators != null) r.symmetryOperators = this.symmetryOperators;
     if (scale != null) r.scale = this.scale.clone();
     if (scaleInverse != null) r.scaleInverse = this.scaleInverse.clone();
     if (origx != null) r.origx = this.origx.clone();
     if (origxInverse != null) r.origxInverse = this.origxInverse.clone();
     r.isOrigxIdentity = this.isOrigxIdentity;
     if (translationVector != null) r.translationVector = this.translationVector.clone();
     if (NCSmatrix != null) r.NCSmatrix = this.NCSmatrix.clone();
     r.cellChecker = this.cellChecker;
    
     return r;
 }
 
 public String SpaceGroupInformation()
 {
     StringBuilder sb = new StringBuilder();
     
     try {
        SpaceGroupInformation sgi = SpaceGroupTableByName.get(name);
        sb.append(this.crystalSystemText());
        sb.append("\n\n");
        sb.append("Space Group : " + sgi.Name);
        sb.append('\n');
        sb.append("Other allowed name : " + sgi.Name2);
        sb.append('\n');
        sb.append("Symmetry operators : " + symmetryOperators.length);
        sb.append('\n');
        for (int i=0; i<symmetryOperators.length; i++)
        {
            sb.append(' ');
            for (int j=0; j<3; j++)
            {
                sb.append(symmetryOperators[i][j]);
                sb.append("  ");
            }
            sb.append('\n');
        }
        
     }
     catch (Exception e)
     {
         return "Space group : " + name + "\nNo information available";
     }
         
     return sb.toString();
 }
 
 public boolean isfCellParametersCorrect()
 {
     return cellChecker.CheckCell(this);
 }
 
 public boolean CorrectCellParameters()
 {
     boolean value = cellChecker.CheckCell(this);
     cellChecker.Correct(this);
     return value;
 }
 
 public void GetSpaceGroupInformation(String Value)
 {
     GetSpaceGroupInformation(SpaceGroupTableByName.get(Value));
 }

 public void GetSpaceGroupInformation(int Value)
 {
     GetSpaceGroupInformation(SpaceGroupTableByNumber.get(Value));
 }

 public void GetSpaceGroupInformation(SpaceGroupInformation sgi)
 {
     symmetryOperators = sgi.Operators;  
     name = sgi.Name;
     number = sgi.CCP4Number;
     cellChecker = CrystalSystemChecker.GetChecker(sgi.System);
 }


 public void ReadFromPDBLines(String[] Lines) 
 {
   for (int i=0; i<Lines.length; i++)
   {
       if (Lines[i].length() > 5)
       { 
            if (Lines[i].subSequence(0, 6).equals("CRYST1")) 
                GetPDBCRYST1Information(Lines[i]);
            if (Lines[i].subSequence(0, 5).equals("ORIGX")) 
                GetPDBORIGXSCALEInformation(Lines[i]);
            if (Lines[i].subSequence(0, 5).equals("SCALE")) 
                GetPDBORIGXSCALEInformation(Lines[i]);
       }
   }
 }
 
 public String WritePDBSpaceGroupLines()
 {
     StringBuilder sb = new StringBuilder();
     
     if (name != null) sb.append(WritePDBCRYSTLine());
     if (origx != null) sb.append(WriteORIGXLines());
     if (scale != null) sb.append(WriteSCALELines());
     
     return sb.toString();
 }
 
 public String WritePDBCRYSTLine()
 {
     //		sb.append(String.format("%-3s ", residueType));
     StringBuilder sb = new StringBuilder();
     
     sb.append("CRYST1");
     sb.append(String.format("%9.3f", a));
     sb.append(String.format("%9.3f", b));
     sb.append(String.format("%9.3f", c));
     sb.append(String.format("%7.2f", alpha));
     sb.append(String.format("%7.2f", beta));
     sb.append(String.format("%7.2f", gamma));
     sb.append(' ');
     sb.append(String.format("%-11s", name));
     sb.append('\n');
     
     return sb.toString();
 }
 
 /*
   * COLUMNS     DATA TYPE        FIELD        DEFINITION
----------------------------------------------------
 1 -  6     Record name      "SCALEn"     n=1, 2, or 3
11 - 20     Real(10.6)       s[n][1]      Sn1
21 - 30     Real(10.6)       s[n][2]      Sn2
31 - 40     Real(10.6)       s[n][3]      Sn3
46 - 55     Real(10.5)       u[n]         Un
  */
 public String WriteORIGXLines()
 {
    return WritePDBMatrix(origx, "ORIGX");
 }
 
 public String WriteSCALELines()
 {
     return WritePDBMatrix(scale, "SCALE");
 }
 
 public String WritePDBMatrix(double[][] matrix, String MName)
 {
     StringBuilder sb = new StringBuilder();
     
     for (int i=0; i<3; i++)
     {
         sb.append(MName + (i+1));
         sb.append("    ");
         sb.append(String.format("%10.6f", matrix[i][0]));
         sb.append(String.format("%10.6f", matrix[i][1]));
         sb.append(String.format("%10.6f", matrix[i][2]));
         sb.append("     ");
         sb.append(String.format("%10.5f", matrix[i][3]));
         sb.append('\n');
     }
     
     return sb.toString();
 }
 
 // Private functions
 /*
  PDB Record Format
COLUMNS      DATA TYPE            FIELD        DEFINITION
----------------------------------------------------------
1 -  6      Record name          "CRYST1"
7 - 15      Real(9.3)            a            a (Angstroms).
16 - 24      Real(9.3)            b            b (Angstroms).
25 - 33      Real(9.3)            c            c (Angstroms).
34 - 40      Real(7.2)            alpha        alpha (degrees).
41 - 47      Real(7.2)            beta         beta (degrees).
48 - 54      Real(7.2)            gamma        gamma (degrees).
56 - 66      LString              sGroup       Space group.
67 - 70      Integer              z            Z value.
  */
 private void GetPDBCRYST1Information(String Line)
 {
    String s = null;
    s = Global.GetSubstring(Line, 6, 15);
    if (s != null) 
    {
        a = Double.parseDouble(s);
        s = Global.GetSubstring(Line, 15, 24);
        if (s != null)
        {
            b = Double.parseDouble(s);
            s = Global.GetSubstring(Line, 24, 33);
            if (s != null)
            {
                c = Double.parseDouble(s);
                s = Global.GetSubstring(Line, 33, 40);
                if (s != null)
                {
                    alpha = Double.parseDouble(s);
                    s = Global.GetSubstring(Line, 40, 47);
                    if (s != null)
                    {
                        beta = Double.parseDouble(s);
                        s = Global.GetSubstring(Line, 47, 54);
                        if (s != null)
                        {
                            gamma = Double.parseDouble(s);
                            s = Global.GetSubstring(Line, 55, 66);
                            if (s != null)
                            {
                                name(s.trim());
                                //if (Line.length() >=65) number = Integer.parseInt(Line.substring(66));
                            }
                        }
                    }
                }
            }
        }
    }
 }
 
 /*
  * COLUMNS     DATA TYPE        FIELD        DEFINITION
----------------------------------------------------
 1 -  6     Record name      "SCALEn"     n=1, 2, or 3
11 - 20     Real(10.6)       s[n][1]      Sn1
21 - 30     Real(10.6)       s[n][2]      Sn2
31 - 40     Real(10.6)       s[n][3]      Sn3
46 - 55     Real(10.5)       u[n]         Un
  */
 private void GetPDBORIGXSCALEInformation(String Line)
 {
	 double[][] Matrix = null;
	 boolean isOrigx = false;
	 if (Line.subSequence(0, 5).equals("ORIGX")) { Matrix = origx; isOrigx = true; } 
	 if (Line.subSequence(0, 5).equals("SCALE")) Matrix = scale;
	 
	 if (Matrix == null)
	 {
		 Matrix = new double[4][4];
		 Matrix[3][0] = 0;
		 Matrix[3][1] = 0;
		 Matrix[3][2] = 0;
		 Matrix[3][3] = 1;
	 }
	 int i = -1;
	 
         if (Line.length() > 6)
         {
         if (Line.substring(5, 6).equals("1")) i = 0;
	 if (Line.substring(5, 6).equals("2")) i = 1;
	 if (Line.substring(5, 6).equals("3")) i = 2;
	 
	 if (i != -1)
	 {
		 try {
                 Matrix[i][0] = Double.parseDouble(Line.substring(10, 20));
		 Matrix[i][1] = Double.parseDouble(Line.substring(20, 30));
		 Matrix[i][2] = Double.parseDouble(Line.substring(30, 40));
		 Matrix[i][3] = Double.parseDouble(Line.substring(45, 55));
                 }
                 catch (Exception e)
                 {
                     
                 }
                 if (isOrigx) origx = Matrix;
                 else scale = Matrix;
		
		 if (i == 2)
		 {
			 if (isOrigx)
			 {
                            origx = Matrix;
                            origxInverse = Global.Inverse4x4Matrix(Matrix);
                            isOrigxIdentity = Global.IsIdentityMatrix(Matrix);
			 }
			 else
			 {
                            scale = Matrix;
                            scaleInverse = Global.Inverse4x4Matrix(Matrix);
			 }
		 }
	 }
         }
     }
}
