package pdb_reader.data;

import pdb_reader.data.spacegroup.SpaceGroup;
import pdb_reader.Global;

public class AtomPDB extends Atom{
	static public AtomPDB AtomPDBclass = new AtomPDB();
	
	static public String[] Headers = new String[] { "ATOM", "HETATM" };
	static public String[] ContinueHeaders = new String[] { "SIGATM", "ANISOU", "SIGUIJ" };
        static public String[] IgnoreHeaders = new String[] { "TER" };
        
	// Static Range Correction functions
	static public final double CoordinateMaximum = 9999.999;
	static public final double CoordinateMinimum = -999.999;
	static private Coordinate CorrectCoordinateRange(Coordinate Value)
	{
            if (Value != null) {
                Value.x(AtomPDB.CorrectCoordinateRange(Value.x()));
		Value.y(AtomPDB.CorrectCoordinateRange(Value.y()));
		Value.z(AtomPDB.CorrectCoordinateRange(Value.z()));
		return Value;
            }
            else return null;
	}

	static private double CorrectCoordinateRange(double Value)
	{
		if (Value > AtomPDB.CoordinateMaximum) return AtomPDB.CoordinateMaximum;
		if (Value < AtomPDB.CoordinateMinimum) return AtomPDB.CoordinateMinimum;
		return Value;
	}
	static public final int UMaximum = 9999999;
	static public final int UMinimum = -999999;
	static private int[] CorrectURange(int[] Value)
	{
            if (Value != null) {
                for (int i=0; i<6; i++) Value[i] = AtomPDB.CorrectURange(Value[i]);
                return Value;
            }
            else return null;
	}
	static private int CorrectURange(int Value)
	{
		if (Value > AtomPDB.UMaximum) return AtomPDB.UMaximum;
		if (Value < AtomPDB.UMinimum) return AtomPDB.UMinimum;
		return Value;
	}
	static public final double BOccpMaximum = 999.99;
	static public final double BOccpMinimum = 0;
	static private double CorrectBOccpRange(double Value)
	{
		if (Value > AtomPDB.BOccpMaximum) return AtomPDB.BOccpMaximum;
		if (Value < AtomPDB.BOccpMinimum) return AtomPDB.BOccpMinimum;
		return Value;
	}
	
        
        public AtomPDB()
	{
            CommonInitializer();
	}
	public AtomPDB(Atom a)
	{
            CommonInitializer();
            CopyFromAtom(a);
	}
	public AtomPDB(String PDB_ATOM_Line, SpaceGroup spacegroup)
	{
            CommonInitializer();
            this.Spacegroup(spacegroup);
            ReadPDBAtomLine(PDB_ATOM_Line);
	}
        
        public void CommonInitializer()
        {
        }
        
	
	// Interface function
	public void CopyFromAtom(Atom Value)
	{
		this.AlternateLocation(Value.alternateLocation);
		if (Value.anisotropicB != null) this.AnisotropicB(Value.anisotropicB.clone());
		this.AtomNumber(Value.atomNumber);
		this.AtomType(Value.atomType);
		this.B_Factor(Value.B);
		this.ChainID(Value.chainID);
		this.Charge(Value.charge);
		this.Element(Value.element);
		this.HetroAtom(Value.hetroAtom);
		this.InsertionCode(Value.insertionCode);
		if (Value.XYZ != null) this.OrthogonalCoordinate(Value.XYZ.clone());
		this.Occupancy(Value.occupancy);
		this.ResidueNumber(Value.residueNumber);
		this.ResidueType(Value.residueType);
		this.SigmaB_Factor(Value.sigmaB);
		this.SigmaOccupancy(Value.sigmaOccupancy);
                this.Spacegroup(Value.spaceGroup);
		if (Value.sigmaU != null) this.SigmaU(Value.sigmaU.clone());
		if (Value.sigmaXYZ != null) this.SigmaCoordinate(Value.sigmaXYZ.clone());
                if (Value.xyz != null) this.FractionalCoordinate(Value.xyz.clone());
	}
	
	// Overriden Interface functions
	public int AtomNumber(int AtomNumber) {	return super.AtomNumber(Global.CorrectRange(AtomNumber, 99999, 0)); }
	public String AtomType(String AtomType) 
	{ 
            if (AtomType != null) if (AtomType.length() > 4) AtomType = AtomType.substring(0, 4);
            return super.AtomType(AtomType);
	}
	public String ResidueType(String ResidueType) 
	{ 
		return super.ResidueType(ResidueType.substring(0, 3)); 
	}
	public int ResidueNumber(int ResidueNumber) { return super.ResidueNumber(Global.CorrectRange(ResidueNumber, 9999, 0)); }
	public Coordinate OrthogonalCoordinate(Coordinate Value) { return super.OrthogonalCoordinate(AtomPDB.CorrectCoordinateRange(Value)); }
	public double X(double X) { return super.X(CorrectCoordinateRange(X)); }
	public double Y(double Y) { return super.Y(CorrectCoordinateRange(Y)); }
	public double Z(double Z) { return super.Z(CorrectCoordinateRange(Z)); }
	public Coordinate SigmaCoordinate(Coordinate Value) { return super.SigmaCoordinate(CorrectCoordinateRange(Value)); }
	public double SigmaX(double X) { return super.SigmaX(CorrectCoordinateRange(X)); }
	public double SigmaY(double Y) { return super.SigmaY(CorrectCoordinateRange(Y)); }
	public double SigmaZ(double Z) { return super.SigmaZ(CorrectCoordinateRange(Z)); }
	public int[] AnisotropicB(int[] AnisotropicB) { return super.AnisotropicB(CorrectURange(AnisotropicB)); }
	public Integer U11(int Value) { return super.U11(CorrectURange(Value)); }
	public Integer U22(int Value) { return super.U22(CorrectURange(Value)); }
	public Integer U33(int Value) { return super.U33(CorrectURange(Value)); }
	public Integer U12(int Value) { return super.U12(CorrectURange(Value)); }
	public Integer U13(int Value) { return super.U13(CorrectURange(Value)); }
	public Integer U23(int Value) { return super.U23(CorrectURange(Value)); }
	public int[] SigmaU(int[] SigmaU) { return super.SigmaU(CorrectURange(SigmaU)); }
	public Integer SigmaU11(int Value) { return super.SigmaU11(CorrectURange(Value)); }
	public Integer SigmaU22(int Value) { return super.SigmaU22(CorrectURange(Value)); }
	public Integer SigmaU33(int Value) { return super.SigmaU33(CorrectURange(Value)); }
	public Integer SigmaU12(int Value) { return super.SigmaU12(CorrectURange(Value)); }
	public Integer SigmaU13(int Value) { return super.SigmaU13(CorrectURange(Value)); }
	public Integer SigmaU23(int Value) { return super.SigmaU23(CorrectURange(Value)); }
	public double B_Factor(double B_Factor) { return super.B_Factor(CorrectBOccpRange(B_Factor)); }
	public double SigmaB_Factor(double SigmaB_Factor) { return super.SigmaB_Factor(CorrectBOccpRange(SigmaB_Factor)); }
	public double Occupancy(double Occupancy) { return super.Occupancy(CorrectBOccpRange(Occupancy)); }
	public double SigmaOccupancy(double SigmaOccupancy) { return super.SigmaOccupancy(CorrectBOccpRange(SigmaOccupancy)); }

	
	// PDB Text Parse functions
	//    Compositie input functions
	public void ReadPDBAtomLine(String Line)
	{
		/* Record Format
		  COLUMNS      DATA TYPE        FIELD      DEFINITION
------------------------------------------------------
 1 -  6      Record name      "ATOM    "
 7 - 11      Integer          serial     Atom serial number.
13 - 16      Atom             name       Atom name.
17           Character        altLoc     Alternate location indicator.
18 - 20      Residue name     resName    Residue name.
22           Character        chainID    Chain identifier.
23 - 26      Integer          resSeq     Residue sequence number.
27           AChar            iCode      Code for insertion of residues.
31 - 38      Real(8.3)        x          Orthogonal coordinates for X in 
                                         Angstroms
39 - 46      Real(8.3)        y          Orthogonal coordinates for Y in 
                                         Angstroms
47 - 54      Real(8.3)        z          Orthogonal coordinates for Z in 
                                         Angstroms
55 - 60      Real(6.2)        occupancy  Occupancy.
61 - 66      Real(6.2)        tempFactor Temperature factor.
77 - 78      LString(2)       element    Element symbol, right-justified.
79 - 80      LString(2)       charge     Charge on the atom.
		*/
            for (int i=0; i<ReadingErrorRecords.length; i++)
                ReadingErrorRecords[i] = false;

            try {
            String AtomCode = Line.substring(0, 6).trim();
            if (AtomCode.equals("HETATM")) hetroAtom = true;
            } catch (Exception e) { ReportReadingError(HETROATOM_INDEX); }
            
            try {
            atomNumber = Integer.parseInt(Line.substring(6, 11).trim());
            } catch (Exception e) { ReportReadingError(Atom.ATOMNUMBER_INDEX); }
               
            try {
            atomType = Line.substring(12, 16).trim();
            } catch (Exception e) { ReportReadingError(Atom.ATOMNAME_INDEX); }
            
            try {
            alternateLocation = Line.charAt(16);
            } catch (Exception e) { ReportReadingError(Atom.ALTERNATELOCATION_INDEX); }
            
            try {
            residueType = Line.substring(17, 20).trim();
            } catch (Exception e) { ReportReadingError(Atom.RESIDUETYPE_INDEX); }
            
            try {
            chainID = Line.charAt(21);
            } catch (Exception e) { ReportReadingError(Atom.CHAINID_INDEX); }
            
            try {
            residueNumber = Integer.parseInt(Line.substring(22, 26).trim());
            } catch (Exception e) { ReportReadingError(Atom.RESIDUENUMBER_INDEX); }
            
            try {
            insertionCode = Line.charAt(26);
            } catch (Exception e) { ReportReadingError(Atom.INSERTIONCODE_INDEX); }
               
            
            Coordinate OC = new Coordinate();
            OrthogonalCoordinate(OC);
            sigmaXYZ = null;
            try {
            OC.x(Double.parseDouble(Line.substring(30, 38).trim()));
            } catch (Exception e) {ReportReadingError(Atom.X_INDEX);}
            try {
            OC.y(Double.parseDouble(Line.substring(38, 46).trim()));
            } catch (Exception e) {ReportReadingError(Atom.Y_INDEX);}
            try {
            OC.z(Double.parseDouble(Line.substring(46, 54).trim()));
            } catch (Exception e) {ReportReadingError(Atom.Z_INDEX);}
            
            try {
                GenerateFractionalCoordinate();
            } catch (Exception e) { }
            
            try {
            occupancy = Double.parseDouble(Line.substring(54, 60).trim());
            } catch (Exception e) {ReportReadingError(Atom.OCCUPANCY_INDEX);}
            
            try {
            B = Double.parseDouble(Line.substring(60, 66).trim());
            } catch (Exception e) {ReportReadingError(Atom.B_INDEX); }            
            
            try {
            try
            {
                element = Line.substring(76, 78).trim();
                charge = Line.substring(78, 80).trim();
            }	
            catch (Exception e)
            {
                element = Line.substring(76).trim();
                charge = null;
            }
            } catch (Exception e1) { ReportReadingError(Atom.ELEMENT_INDEX);
                                    ReportReadingError(Atom.CHARGE_INDEX);}
            
            anisotropicB = null;
	
            sigmaU = null;
	}
	
	public void AppendPDBInformation(String Line)
	{
		ReadPDBAnisouLine(Line);
		ReadPDBSigatmLine(Line);
		ReadPDBSiguijLine(Line);
	}
	
	public void ReadPDBAnisouLine(String Line)
	{
	/*
	 * COLUMNS     DATA TYPE         FIELD          DEFINITION
--------------------------------------------------------
 1 - 6      Record name       "ANISOU"
 7 - 11     Integer           serial         Atom serial number.
13 - 16     Atom              name           Atom name.
17          Character         altLoc         Alternate location indicator
18 - 20     Residue name      resName        Residue name.
22          Character         chainID        Chain identifier.
23 - 26     Integer           resSeq         Residue sequence number.
27          AChar             iCode          Insertion code.
29 - 35     Integer           u[0][0]        U(1,1)
36 - 42     Integer           u[1][1]        U(2,2)
43 - 49     Integer           u[2][2]        U(3,3)
50 - 56     Integer           u[0][1]        U(1,2)
57 - 63     Integer           u[0][2]        U(1,3)
64 - 70     Integer           u[1][2]        U(2,3)
77 - 78     LString(2)        element        Element symbol, right-justified.
79 - 80     LString(2)        charge         Charge on the atom.
	 */
                if (Line.substring(0, 6).equals("ANISOU"))
		{
			if (CheckRecordInformation(Line))
			{
				anisotropicB = new int[6];
				                          
				try {		
                                anisotropicB[0] = Integer.parseInt(Line.substring(28, 35).trim());
                                } catch (Exception e) { ReportReadingError(Atom.U11_INDEX); }
                                
                                try {        
                                anisotropicB[1] = Integer.parseInt(Line.substring(35, 42).trim());
                                } catch (Exception e) { ReportReadingError(Atom.U22_INDEX); }
                                
                                try {
                                anisotropicB[2] = Integer.parseInt(Line.substring(42, 49).trim());
				} catch (Exception e) { ReportReadingError(Atom.U33_INDEX); }
                                
                                try {
                                anisotropicB[3] = Integer.parseInt(Line.substring(49, 56).trim());
                                } catch (Exception e) { ReportReadingError(Atom.U12_INDEX); }        
                                                        
                                try {
                                anisotropicB[4] = Integer.parseInt(Line.substring(56, 63).trim());
				} catch (Exception e) { ReportReadingError(Atom.U13_INDEX); }
                                
                                try {
                                anisotropicB[5] = Integer.parseInt(Line.substring(63, 70).trim());
				} catch (Exception e) { ReportReadingError(Atom.U23_INDEX); }                          
			}
		}
	}
	
	public void ReadPDBSigatmLine(String Line)
	{
		if (Line.substring(0, 6).equals("SIGATM"))
		{
			if (CheckRecordInformation(Line))
			{
				sigmaXYZ = new Coordinate();
				
                                try {
                                sigmaXYZ.x(Double.parseDouble(Line.substring(30, 38).trim()));
				} catch (Exception e) { ReportReadingError(Atom.SIGMAX_INDEX); }
                                        
                                try {
                                sigmaXYZ.y(Double.parseDouble(Line.substring(38, 46).trim()));
				} catch (Exception e) { ReportReadingError(Atom.SIGMAY_INDEX); }
                                
                                try {
                                sigmaXYZ.z(Double.parseDouble(Line.substring(46, 54).trim()));
				} catch (Exception e) { ReportReadingError(Atom.SIGMAZ_INDEX); }
                                
                                try {
                                sigmaOccupancy = Double.parseDouble(Line.substring(54, 60).trim());
				} catch (Exception e) { ReportReadingError(Atom.SIGMAOCCUPANCY_INDEX); }
                                
                                try {
                                sigmaB = Double.parseDouble(Line.substring(60, 66).trim());
                                } catch (Exception e) { ReportReadingError(Atom.SIGMAB_INDEX); }
			}
		}
	}
	
	public void ReadPDBSiguijLine(String Line)
	{
		if (Line.substring(0, 6).equals("SIGUIJ"))
		{
			if (CheckRecordInformation(Line))
			{
				sigmaU = new int[6];
				                          
				try {		
                                sigmaU[0] = Integer.parseInt(Line.substring(28, 35).trim());
                                } catch (Exception e) { ReportReadingError(Atom.SIGMAU11_INDEX); }
                                
                                try {        
                                sigmaU[1] = Integer.parseInt(Line.substring(35, 42).trim());
                                } catch (Exception e) { ReportReadingError(Atom.SIGMAU22_INDEX); }
                                
                                try {
                                sigmaU[2] = Integer.parseInt(Line.substring(42, 49).trim());
				} catch (Exception e) { ReportReadingError(Atom.SIGMAU33_INDEX); }
                                
                                try {
                                sigmaU[3] = Integer.parseInt(Line.substring(49, 56).trim());
                                } catch (Exception e) { ReportReadingError(Atom.SIGMAU12_INDEX); }        
                                                        
                                try {
                                sigmaU[4] = Integer.parseInt(Line.substring(56, 63).trim());
				} catch (Exception e) { ReportReadingError(Atom.SIGMAU13_INDEX); }
                                
                                try {
                                sigmaU[5] = Integer.parseInt(Line.substring(63, 70).trim());
				} catch (Exception e) { ReportReadingError(Atom.SIGMAU23_INDEX); }                          
			}
		}

	}
	
	//   Composite output functions
	public String WritePDBLine()
	{
		StringBuilder sb = new StringBuilder();
		
		String AtomLine = WritePDBAtomLine();
		if (AtomLine == null) return null;
		else sb.append(AtomLine);
		sb.append('\n');
		
		String SigatmLine = WritePDBSigatmLine();
		if (SigatmLine != null) sb.append(SigatmLine + '\n');

		String AnisouLine = WritePDBAnisouLine();
		if (AnisouLine != null) sb.append(AnisouLine + '\n');

		String SiguijLine = WritePDBSiguijLine();
		if (SiguijLine != null) sb.append(SiguijLine + '\n');

		return sb.toString();
	}
	
	public String WritePDBAtomLine ()
	{
		if ((atomType != null) && (residueType != null) && (XYZ != null))
		{
			StringBuilder sb = new StringBuilder();
		
			if (hetroAtom) sb.append(String.format("%-6s", Headers[1]));
			//else sb.append(PDBHeaders[0] + "  ");
			else sb.append(String.format("%-6s", Headers[0]));
			
			sb.append(this.WritePDBCommonInformation());
			sb.append(this.WritePDBAtomCoordinate());
			sb.append(this.WritePDBCommonElementInformation(sb.length()));
			
			return sb.toString();
		}
		else return null;
	}
	
	public String WritePDBSigatmLine()
	{
		if ((atomType != null) && (residueType != null) && (sigmaXYZ != null))
		{
			StringBuilder sb = new StringBuilder();
			sb.append(AtomPDB.ContinueHeaders[0]);
			sb.append(this.WritePDBCommonInformation());
			sb.append(this.WritePDBCoordinate(sigmaXYZ, sigmaOccupancy, sigmaB));
			sb.append(this.WritePDBCommonElementInformation(sb.length()));
			return sb.toString();
		}
		else return null;
	}
	
	public String WritePDBAnisouLine()
	{
		if ((atomType != null) && (residueType != null) && (anisotropicB != null))
		{
			StringBuilder sb = new StringBuilder();
			
			sb.append(AtomPDB.ContinueHeaders[1]);
			sb.append(this.WritePDBCommonInformation());
			sb.append(this.WritePDBAnistropicB(anisotropicB));
			sb.append(this.WritePDBCommonElementInformation(sb.length()));
			
			return sb.toString();
		}
		return null;
	}
	
	public String WritePDBSiguijLine()
	{
		if ((atomType != null) && (residueType != null) && (sigmaU != null))
		{
			StringBuilder sb = new StringBuilder();
			
			sb.append(AtomPDB.ContinueHeaders[2]);
			sb.append(this.WritePDBCommonInformation());
			sb.append(this.WritePDBAnistropicB(sigmaU));
			sb.append(this.WritePDBCommonElementInformation(sb.length()));
			
			return sb.toString();
		}
		return null;
	}
	
	// Private functions
	private boolean CheckRecordInformation(String Line)
	{
		if (atomNumber != Integer.parseInt(Line.substring(6, 11).trim())) return false;
		if (!atomType.equals(Line.substring(12, 16).trim())) return false;
		if (alternateLocation != Line.charAt(16)) return false;
		if (!residueType.equals(Line.substring(17, 20).trim())) return false;
		if (chainID != Line.charAt(21)) return false;
		if (residueNumber != Integer.parseInt(Line.substring(22, 26).trim())) return false;
		if (insertionCode != Line.charAt(26)) return false;
		return true;
	}
	
	private String WritePDBCommonInformation()
	{
		StringBuilder sb = new StringBuilder();
		sb.append(String.format("%5d", atomNumber));
		if (atomType.length() > 3) sb.append(String.format(" %-4s", atomType));
		else sb.append(String.format("  %-3s", atomType));
		sb.append(alternateLocation);
		sb.append(String.format("%-3s ", residueType));
		sb.append(chainID);
		sb.append(String.format("%4d", residueNumber));
		sb.append(insertionCode);
		return sb.toString();
	}
	
	private String WritePDBCommonElementInformation(int currentlength)
	{
		StringBuilder sb = new StringBuilder();
		
		for (int i=currentlength; i < 77; i++)
			sb.append(' ');
		sb.append(String.format("%-2s", element));
		if (charge != null) sb.append(String.format("%-2s", charge));
		
		return sb.toString();
	}
	
	private String WritePDBAtomCoordinate()
	{
		return WritePDBCoordinate(XYZ, occupancy, B);
	}
	
	private String WritePDBCoordinate(Coordinate cin, double occup, double Bin)
	{
		StringBuilder sb = new StringBuilder();
		sb.append("   ");
		sb.append(String.format("%8.3f", cin.x()));
		sb.append(String.format("%8.3f", cin.y()));
		sb.append(String.format("%8.3f", cin.z()));
		sb.append(String.format("%6.2f", occup));
		sb.append(String.format("%6.2f", Bin));
		return sb.toString();
	}
	
	private String WritePDBAnistropicB(int[] ab)
	{
		StringBuilder sb = new StringBuilder();
		
		sb.append(' ');
		sb.append(String.format("%7d", ab[0]));
		sb.append(String.format("%7d", ab[1]));
		sb.append(String.format("%7d", ab[2]));
		sb.append(String.format("%7d", ab[3]));
		sb.append(String.format("%7d", ab[4]));
		sb.append(String.format("%7d", ab[5]));
	
		return sb.toString();
	}
        
        public String WritePDBTerLine()
        {
            /*COLUMNS     DATA TYPE         FIELD           DEFINITION
------------------------------------------------------
 1 - 6      Record name       "TER     "
 7 - 11     Integer           serial          Serial number.
18 - 20     Residue name      resName         Residue name.
22          Character         chainID         Chain identifier.
23 - 26     Integer           resSeq          Residue sequence 
                                              number.
27          AChar             iCode           Insertion code.
*/
            
            StringBuilder sb = new StringBuilder();
            
            sb.append("TER   ");
            sb.append(String.format("%5d", atomNumber));
            sb.append("      ");
            sb.append(String.format("%-3s ", residueType));
            sb.append(chainID);
            sb.append(String.format("%4d", residueNumber));
            sb.append(insertionCode);
            sb.append('\n');
            
            return sb.toString();
        }
        
        private void ReportReadingError()
        {
            ReadingErrorRecords[Atom.TOTAL_NUMBER_OF_INDEX] = true;
        }
        private void ReportReadingError(int DataIndex)
        {
            ReadingErrorRecords[DataIndex] = true;
            ReportReadingError();
        }
        
}
