/*
 *  bayreg/grid.c 
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:47 $, $Version$
 *  
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */
 
#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_bayesian_regression.h>

gdl_bayreg_grid_point *
gdl_bayreg_grid_point_clone (const gdl_bayreg_grid_point * p)
{
	if (p)
	{
		gdl_bayreg_grid_point * c;
		
		c = GDL_MALLOC (gdl_bayreg_grid_point, 1);
		
		memcpy (c, p, sizeof(gdl_bayreg_grid_point));
		
		return c;
	}
	return 0;
}

gdl_bayreg_grid_point *
gdl_bayreg_grid_point_fread (FILE * stream)
{
	if (stream)
	{
		int status;
		gdl_bayreg_grid_point * p;
		
		p = GDL_MALLOC (gdl_bayreg_grid_point, 1);
		
		status = fread (&p->sigmaa, sizeof(double), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		status = fread (&p->sigmad, sizeof(double), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		
		return p;
	}
	return 0;	
}

int
gdl_bayreg_grid_point_fwrite (FILE * stream, const gdl_bayreg_grid_point * p)
{
	if (stream && p)
	{
		int status;
		
		status = fwrite (&p->sigmaa, sizeof(double), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		status = fwrite (&p->sigmad, sizeof(double), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		
		return GDL_SUCCESS;
	}
	return GDL_EINVAL;	
}

gdl_bayreg_grid *
gdl_bayreg_grid_alloc (void)
{
	gdl_bayreg_grid * g;
	
	g = GDL_CALLOC (gdl_bayreg_grid, 1);
	
	return g;	
}

void
gdl_bayreg_grid_free (gdl_bayreg_grid * g)
{
	if (g)
	{
		size_t i;
		for (i = 0; i < g->size; i++)
			GDL_FREE (g->points[i]);
		GDL_FREE (g->points);
		GDL_FREE (g->weight);
		GDL_FREE (g);	
	}
}

gdl_bayreg_grid *
gdl_bayreg_grid_clone (const gdl_bayreg_grid * g)
{
	if (g)
	{
		size_t i;
		gdl_bayreg_grid * c;
		
		c = gdl_bayreg_grid_alloc ();
		
		c->size   = g->size;
		c->weight = GDL_MALLOC (double, g->size);
		memcpy (c->weight, g->weight, sizeof(double)*g->size);
		
		c->points = GDL_MALLOC (gdl_bayreg_grid_point *, g->size);
		for (i = 0; i < c->size; i++)
		{
			c->points[i] = gdl_bayreg_grid_point_clone (g->points[i]); 	
		}
		
		return c;		
	}
	return 0;
}

void
gdl_bayreg_grid_add (gdl_bayreg_grid * g, const gdl_bayreg_grid_point * p, const double w)
{
	if (p)
	{
		gdl_bayreg_grid_point ** tmp = GDL_MALLOC (gdl_bayreg_grid_point *, g->size+1);
		memcpy (tmp, g->points, sizeof(gdl_bayreg_grid_point *)*g->size);
		GDL_FREE (g->points);
		g->points = tmp;
		double * dtmp = GDL_MALLOC (double, g->size+1);
		memcpy (dtmp, g->weight, sizeof(double)*g->size);
		GDL_FREE (g->weight);
		g->weight = dtmp;
	}
	else
	{
		g->points = GDL_MALLOC (gdl_bayreg_grid_point *, 1);	
		g->weight = GDL_MALLOC (double, 1);
		//g->maxcount = GDL_CALLOC (size_t, 1);
	}
	g->points[g->size] = gdl_bayreg_grid_point_clone (p);
	g->weight[g->size] = w;
	g->size++;	
}

gdl_bayreg_grid *
gdl_bayreg_grid_fread (FILE * stream)
{
	if (stream)
	{
		int status;
		size_t i;
		gdl_bayreg_grid * g;
		
		g = gdl_bayreg_grid_alloc ();
		
		status = fread (&g->size, sizeof(size_t), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		
		g->points = GDL_MALLOC (gdl_bayreg_grid_point *, g->size);	
		g->weight = GDL_MALLOC (double, g->size);
		
		status = fread (g->weight, sizeof(double), g->size, stream);
		GDL_FREAD_STATUS (status, g->size);
		
		for (i = 0; i < g->size; i++)
		{
			g->points[i] = gdl_bayreg_grid_point_fread (stream);
			GDL_FREAD_STATUS (g->points[i]!=0, 1);	
		}
		
		return g;
	}
	return 0;	
}

int
gdl_bayreg_grid_fwrite (FILE * stream, const gdl_bayreg_grid * g)
{
	if (stream && g)
	{
		int status;
		size_t i;
		
		status = fwrite (&g->size, sizeof(size_t), 1, stream);
		GDL_FWRITE_STATUS (status, 1);
		
		status = fwrite (g->weight, sizeof(double), g->size, stream);
		GDL_FWRITE_STATUS (status, g->size);
		
		for (i = 0; i < g->size; i++)
		{
			status = gdl_bayreg_grid_point_fwrite (stream, g->points[i]);
			GDL_FREAD_STATUS (status, GDL_SUCCESS);	
		}
		
		return GDL_SUCCESS;
	}
	return GDL_EINVAL;		
}

