/*
 *  bayreg/resdom.c 
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:47 $, $Version$
 *  
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */

#include <stdio.h>
 
#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_bayesian_regression.h>

gdl_bayreg_resdom_model *
gdl_bayreg_resdom_model_alloc (const double w, const double q, gdl_bayreg_grid * g)
{
	gdl_bayreg_resdom_model * m;
	
	m = GDL_MALLOC (gdl_bayreg_resdom_model, 1);
	
	m->w    = w;
	m->pres = q;
	m->pdom = 1-q;
	m->g    = g;
	
	return m;	
}

void
gdl_bayreg_resdom_model_free (gdl_bayreg_resdom_model * m)
{
	if (m)
	{
		gdl_bayreg_grid_free (m->g);
		GDL_FREE (m);
	}	
}

gdl_bayreg_resdom_model *
gdl_bayreg_resdom_model_fread (FILE * stream)
{
	if (stream)
	{
		int status;
		double w, q;
		gdl_bayreg_resdom_model * m;
		gdl_bayreg_grid * g;
		
		status = fread (&w, sizeof(double), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		status = fread (&q, sizeof(double), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		g = gdl_bayreg_grid_fread (stream);
		GDL_FREAD_STATUS (g!=0, 1);
		
		m = gdl_bayreg_resdom_model_alloc (w, q, g);
		
		return m;
	}
	return 0;	
}

gdl_bayreg_resdom_model *
gdl_bayreg_resdom_model_fscanf (FILE * stream)
{
	size_t i,j,n;
	gdl_string * tok, * line = 0;
	double w, q, wp;
	gdl_bayreg_grid_point p;
	gdl_bayreg_grid * g = 0;
	gdl_bayreg_resdom_model * m = 0;
	long offset = ftell (stream);
	
	while (gdl_getline (&line, &n, stream) != -1)
	{
	   for(i = 0;i < n && isspace(line[i]); i++);
		if (line[i]!='#')
		{
			i = j = 0;
			tok = gdl_string_next_token (line, n, &i, &j);
			if (!strcmp(tok,"<r>"))
			{
				gdl_string_free (tok);
				tok = gdl_string_next_token (line, n, &i, &j);
				w   = (double)atof(tok);
				gdl_string_free (tok);
				tok = gdl_string_next_token (line, n, &i, &j);
				q   = (double)atof(tok);
				gdl_string_free (tok);
				gdl_string_free (line);line=0;
				
				g = gdl_bayreg_grid_alloc ();
				
				while (gdl_getline (&line, &n, stream) != -1)
				{
					i = j = 0;
					tok = gdl_string_next_token (line, n, &i, &j);
					if (!strcmp(tok,"</r>"))
					{
						gdl_string_free (tok);
						gdl_string_free (line);line=0;
						break;
					}
					wp = (double)atof(tok);
					gdl_string_free (tok);
					tok = gdl_string_next_token (line, n, &i, &j);
					p.sigmaa = (double)atof(tok);
					gdl_string_free (tok);
					p.sigmad = 0;
					gdl_bayreg_grid_add (g, &p, wp);
					gdl_string_free (line);line=0;
				}
				m = gdl_bayreg_resdom_model_alloc (w, q, g);
				break;
			}
			else
			{
				gdl_string_free (tok);
				gdl_string_free (line);line=0;
				fseek (stream, offset, SEEK_SET);
				break;
			}
		}
		gdl_string_free (line);line=0;
	}
	
	return m;
}

int
gdl_bayreg_resdom_model_fwrite (FILE * stream, const gdl_bayreg_resdom_model * m)
{
	if (stream && m)
	{
		int status;
		double q = (m->pres)/(m->pres+m->pdom);
		
		status = fwrite (&m->w, sizeof(double), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		status = fwrite (&q, sizeof(double), 1, stream);
		GDL_FREAD_STATUS (status, 1);
		status = gdl_bayreg_grid_fwrite (stream, m->g);
		GDL_FREAD_STATUS (status, GDL_SUCCESS);
		
		return GDL_SUCCESS;
	}
	return GDL_EINVAL;
}
