/*
 *  bayreg/work.c 
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:47 $, $Version$
 *  
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */
#include <math.h>

#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_string.h>
#include <gdl/gdl_matrix.h>
#include <gdl/gdl_vector.h>
#include <gdl/gdl_blas.h>
#include <gdl/gdl_cblas.h>
#include <gdl/gdl_math.h>
#include <gdl/gdl_rng.h>
#include <gdl/gdl_randist.h>
#include <gdl/gdl_specfunc.h>
#include <gdl/gdl_sort_double.h>
#include <gdl/gdl_statistics_double.h>
#include <gdl/gdl_snp_annotation.h>
#include <gdl/gdl_bayesian_regression.h>

gdl_bayreg_workspace *
gdl_bayreg_workspace_alloc (const size_t N, const size_t P)
{
	size_t i, j;
	gdl_bayreg_workspace * b;	
	
	b = GDL_CALLOC (gdl_bayreg_workspace, 1);
	
	b->P = P;
	b->N = N;	
		
	b->X   = gdl_matrix_alloc (N, P+2);
	b->Q   = gdl_matrix_alloc (P+2, P+2);
	b->QSI = gdl_matrix_alloc (P+2, P+2);
	
	b->y  = gdl_vector_alloc (N);
	b->e  = gdl_vector_alloc (N);
	b->S  = gdl_vector_alloc (P+2);
	b->yy = gdl_vector_alloc (P+2);
	b->D  = gdl_vector_alloc (P+2);
	b->b  = gdl_vector_alloc (P+2);
 	b->xt = gdl_vector_alloc (P+2);
 	
 	for (i = 0; i < b->X->size1; i++)
 	{
 		gdl_matrix_set (b->X, i, 0, 1);
 	}
 	
	return b;
}

gdl_bayreg_workspace *
gdl_bayreg_workspace_forward_alloc (const size_t N, const size_t P)
{
	size_t i, j;
	gdl_bayreg_workspace * b;	
	
	b = GDL_CALLOC (gdl_bayreg_workspace, 1);
	
	b->P = P;
	b->N = N;
		
	b->Xm   = gdl_matrix_alloc (N, P+3);
	b->sigm = gdl_vector_alloc (P+2);
	b->Q    = gdl_matrix_alloc (P+3, P+3);
	b->QSI  = gdl_matrix_alloc (P+3, P+3);
	
	b->y  = gdl_vector_alloc (N);
	b->e  = gdl_vector_alloc (N);
	b->S  = gdl_vector_alloc (P+3);
	b->yy = gdl_vector_alloc (P+3);
	b->D  = gdl_vector_alloc (P+3);
	b->b  = gdl_vector_alloc (P+3);
 	b->xt = gdl_vector_alloc (P+3);
 	
 	for (i = 0; i < b->X->size1; i++)
 	{
 		gdl_matrix_set (b->Xm, i, 0, 1);
 	}
 	
	return b;
}

void
gdl_bayreg_workspace_free (gdl_bayreg_workspace * b)
{
	if (b)
	{
		gdl_matrix_free (b->X);
		gdl_matrix_free (b->Y);
		gdl_matrix_free (b->Q);
		gdl_matrix_free (b->QSI);
		gdl_matrix_free (b->invOmega);
		gdl_vector_free (b->y);
		gdl_vector_free (b->e);
		gdl_vector_free (b->yy);
		gdl_vector_free (b->S);
		gdl_vector_free (b->D);
		gdl_vector_free (b->b);
		gdl_vector_free (b->xt);
		GDL_FREE (b->pidx);
		GDL_FREE (b);	
	}	
}

#define VECTOR_SET(X,i,y)(*(X->data+i*X->stride)=y)
#define VECTOR_GET(X,i)(*(X->data+i*X->stride))
#define MATRIX_SET(X,i,j,y)(*(X->data+(i*X->tda+j))=y)
#define MATRIX_GET(X,i,j)(*(X->data+(i*X->tda+j)))

#include "../linalg/givens.c"
#include "../linalg/svdstep.c"

static int
gdl_bayreg_workspace_svd (gdl_matrix * A, gdl_matrix * V, gdl_vector * S, gdl_vector * work)
{
	size_t a, b, i, j;
  
  const size_t M = A->size1;
  const size_t N = A->size2;
  const size_t K = GDL_MIN (M, N);

  {
    gdl_vector_view f = gdl_vector_subvector (work, 0, K - 1);
    
    /* bidiagonalize matrix A, unpack A into U S V */
    
    gdl_linalg_bidiag_decomp (A, S, &f.vector);
    gdl_linalg_bidiag_unpack2 (A, S, &f.vector, V);
    
    /* apply reduction steps to B=(S,Sd) */
    
    chop_small_elements (S, &f.vector);
    
    /* Progressively reduce the matrix until it is diagonal */
    
    b = N - 1;
    
    while (b > 0)
      {
        double fbm1 = VECTOR_GET ((&f.vector), b - 1);

        if (fbm1 == 0.0 || gdl_isnan (fbm1))
          {
            b--;
            continue;
          }
        
        /* Find the largest unreduced block (a,b) starting from b
           and working backwards */
        
        a = b - 1;
        
        while (a > 0)
          {
            double fam1 = VECTOR_GET ((&f.vector), a - 1);

            if (fam1 == 0.0 || gdl_isnan (fam1))
              {
                break;
              }
            
            a--;
          }
        
        {
          const size_t n_block = b - a + 1;
          gdl_vector_view S_block = gdl_vector_subvector (S, a, n_block);
          gdl_vector_view f_block = gdl_vector_subvector (&f.vector, a, n_block - 1);
          
          gdl_matrix_view U_block =
            gdl_matrix_submatrix (A, 0, a, A->size1, n_block);
          gdl_matrix_view V_block =
            gdl_matrix_submatrix (V, 0, a, V->size1, n_block);
          
          qrstep (&S_block.vector, &f_block.vector, &U_block.matrix, &V_block.matrix);
          
          /* remove any small off-diagonal elements */
          
          chop_small_elements (&S_block.vector, &f_block.vector);
        }
      }
  }
  /* Make singular values positive by reflections if necessary */
  
  for (j = 0; j < K; j++)
    {
      double Sj = VECTOR_GET (S, j);
      
      if (Sj < 0.0)
        {
          for (i = 0; i < N; i++)
            {
              double Vij = MATRIX_GET (V, i, j);
              MATRIX_SET (V, i, j, (-Vij));
            }
          
          VECTOR_SET (S, j, -Sj);
        }
    }

  return GDL_SUCCESS;	
}

static int
gdl_bayreg_workspace_solve (gdl_bayreg_workspace * bayes)
{
   const size_t n = bayes->invOmega->size1;
   const size_t p = bayes->invOmega->size2;

   size_t i, j, p_eff;
   double alpha0;

   /* Decompose A into U S Q^T */
   
   gdl_bayreg_workspace_svd (bayes->invOmega, bayes->Q, bayes->S, bayes->xt);
   
   /* Solve y = A c for c */

   gdl_blas_dgemv (CblasTrans, 1.0, bayes->invOmega, bayes->yy, 0.0, bayes->xt);

   bayes->det = 1;
   
    for (j = 0; j < p; j++)
    {
      gdl_vector_view column = gdl_matrix_column (bayes->Q, j);
      double alpha = VECTOR_GET(bayes->S, j);
      bayes->det *= alpha;
      alpha = 1.0 / alpha;
      gdl_vector_scale (&column.vector, alpha);
    }

    gdl_blas_dgemv (CblasNoTrans, 1.0, bayes->Q, bayes->xt, 0.0, bayes->b);

    return GDL_SUCCESS;
}

double
gdl_bayreg_single_adddom_bf (gdl_bayreg_workspace * bayes, const double sigmaa, const double sigmad)
{
	const size_t N = bayes->X->size1;
	
	size_t i, j;
	double e0, e1, t, s, s2, u, v, bf;
	
	gdl_matrix_free (bayes->invOmega);
	
	bayes->invOmega = gdl_matrix_calloc (3, 3);
	
	MATRIX_SET (bayes->invOmega, 1, 1, 1/(sigmaa*sigmaa));
	MATRIX_SET (bayes->invOmega, 2, 2, 1/(sigmad*sigmad));

	bayes->e->size = N;
	bayes->b->size = bayes->D->size = bayes->yy->size = bayes->xt->size = bayes->S->size = bayes->Q->size1 = bayes->Q->size2 = bayes->X->size2 = 3;
	
	gdl_blas_dgemm (CblasTrans, CblasNoTrans, 1, bayes->X, bayes->X, 1, bayes->invOmega);
	gdl_blas_dgemv (CblasTrans, 1, bayes->X, bayes->y, 0, bayes->yy);
	
	gdl_bayreg_workspace_solve (bayes);
	
	e1 = s = s2 = 0;
	for (i = 0; i < N; i++)
	{
		t   = VECTOR_GET (bayes->y, i);
		s2 += t*t;
		s  += t;
		v   = t;
		for (j = 0; j < 3; j++)
		{
			v -= VECTOR_GET (bayes->b, j) * MATRIX_GET (bayes->X, i, j);
		}
		gdl_vector_set (bayes->e, i, v);
		e1 += v*t;
	}
	e0 = s2-s*s/N;
	
	bf  = -0.5*log(bayes->det);
	bf += 0.5*log(N);
	bf -= log (sigmaa);
	bf -= log (sigmad);
	bf -= 0.5*N*(log(e1)-log(e0));
	
	return exp(bf);
}

double
gdl_bayreg_multiple_adddom_bf (gdl_bayreg_workspace * bayes)
{
	const size_t N = bayes->Xm->size1;
	const size_t M = bayes->Xm->size2;
	
	size_t i, j;
	double e0, e1, t, s, s2, u, v, bf, lst=0;
	
	gdl_matrix_free (bayes->invOmega);
	
	bayes->invOmega = gdl_matrix_calloc (M, M);
	
	for(i = 1; i < M; i++)
	{
		t = VECTOR_GET (bayes->sigm, i-1);
		lst += log (t);
		MATRIX_SET (bayes->invOmega, i, i, 1.0/(t*t));
	}

	bayes->e->size = N;
	bayes->b->size = bayes->D->size = bayes->yy->size = bayes->xt->size = bayes->S->size = bayes->Q->size1 = bayes->Q->size2 = M;
	
	gdl_blas_dgemm (CblasTrans, CblasNoTrans, 1, bayes->Xm, bayes->Xm, 1, bayes->invOmega);
	gdl_blas_dgemv (CblasTrans, 1, bayes->Xm, bayes->y, 0, bayes->yy);
	
	gdl_bayreg_workspace_solve (bayes);
	
	e1 = s = s2 = 0;
	for (i = 0; i < N; i++)
	{
		t   = VECTOR_GET (bayes->y, i);
		s2 += t*t;
		s  += t;
		v   = t;
		for (j = 0; j < M; j++)
		{
			v -= VECTOR_GET (bayes->b, j) * MATRIX_GET (bayes->Xm, i, j);
		}
		gdl_vector_set (bayes->e, i, v);
		e1 += v*t;
	}
	e0 = s2-s*s/N;
	
	bf  = -0.5*log(bayes->det);
	bf += 0.5*log(N);
	bf -= lst;
	bf -= 0.5*N*(log(e1)-log(e0));
	
	return exp(bf);
}

double
gdl_bayreg_single_puradd_bf (gdl_bayreg_workspace * bayes, const double sigmaa)
{
	const size_t N = bayes->X->size1;
	const size_t M = bayes->X->size2;
	
	size_t i, j;
	double e0, e1, t, s, s2, u, v, bf;
	
	gdl_matrix_free (bayes->invOmega);
	
	bayes->invOmega = gdl_matrix_calloc (2, 2);
	
	MATRIX_SET (bayes->invOmega, 1, 1, 1/(sigmaa*sigmaa));
	
	bayes->e->size = N;
	bayes->b->size = bayes->D->size = bayes->yy->size = bayes->xt->size = bayes->S->size = bayes->Q->size1 = bayes->Q->size2 = bayes->X->size2 = 2;
	
	gdl_blas_dgemm (CblasTrans, CblasNoTrans, 1, bayes->X, bayes->X, 1, bayes->invOmega);
	gdl_blas_dgemv (CblasTrans, 1, bayes->X, bayes->y, 0, bayes->yy);
	
	gdl_bayreg_workspace_solve (bayes);
	
	e1 = s = s2 = 0;
	for (i = 0; i < N; i++)
	{
		t   = VECTOR_GET (bayes->y, i);
		s2 += t*t;
		s  += t;
		v   = t;
		for (j = 0; j < 2; j++)
		{
			v -= VECTOR_GET (bayes->b, j) * MATRIX_GET (bayes->X, i, j);
		}
		//fprintf(stderr, "%d %g %g (%g %g)\n", i, t, v, VECTOR_GET (bayes->b, 0), VECTOR_GET (bayes->b, 1));
		//v += t;
		gdl_vector_set (bayes->e, i, v);
		e1 += v*t;
	}
	e0 = s2-s*s/N;
	
	bf  = -0.5*log(bayes->det);
	bf += 0.5*log(N);
	bf -= log (sigmaa);
	bf -= 0.5*N*(log(e1)-log(e0));
	
	bayes->X->size2 = M;
	
	return exp(bf);
}

double
gdl_bayreg_multiple_puradd_bf (gdl_bayreg_workspace * bayes)
{
	const size_t N = bayes->Xm->size1;
	const size_t M = bayes->Xm->size2-1;
	
	size_t i, j;
	double e0, e1, t, s, s2, u, v, bf, lst=0;
	
	gdl_matrix_free (bayes->invOmega);
	
	bayes->invOmega = gdl_matrix_calloc (M, M);
	
	for(i = 1; i < M; i++)
	{
		t = VECTOR_GET (bayes->sigm, i-1);
		lst += log (t);
		MATRIX_SET (bayes->invOmega, i, i, 1/(t*t));
	}
	
	bayes->e->size = N;
	bayes->Xm->size2 = bayes->b->size = bayes->D->size = bayes->yy->size = bayes->xt->size = bayes->S->size = bayes->Q->size1 = bayes->Q->size2 = M;
	
	gdl_blas_dgemm (CblasTrans, CblasNoTrans, 1, bayes->Xm, bayes->Xm, 1, bayes->invOmega);
	gdl_blas_dgemv (CblasTrans, 1, bayes->Xm, bayes->y, 0, bayes->yy);
	
	gdl_bayreg_workspace_solve (bayes);
	
	e1 = s = s2 = 0;
	for (i = 0; i < N; i++)
	{
		t   = VECTOR_GET (bayes->y, i);
		s2 += t*t;
		s  += t;
		v   = t;
		for (j = 0; j < M; j++)
		{
			v -= VECTOR_GET (bayes->b, j) * MATRIX_GET (bayes->Xm, i, j);
		}
		//fprintf(stderr, "%d %g %g (%g %g)\n", i, t, v, VECTOR_GET (bayes->b, 0), VECTOR_GET (bayes->b, 1));
		//v += t;
		gdl_vector_set (bayes->e, i, v);
		e1 += v*t;
	}
	e0 = s2-s*s/N;
	
	bf  = -0.5*log(bayes->det);
	bf += 0.5*log(N);
	bf -= lst;
	bf -= 0.5*N*(log(e1)-log(e0));
	
	bayes->Xm->size2 = M+1;
	
	return exp(bf);
}

#undef VECTOR_SET
#undef VECTOR_GET
#undef MATRIX_SET
#undef MATRIX_GET
