/*  
 *  dna/gdl_dna_reader.h
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:52 $, $Version$
 *
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */
 
#ifndef __GDL_DNA_READER_H__
#define __GDL_DNA_READER_H__

#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_gpoint.h>
#include <gdl/gdl_string.h>
#include <gdl/gdl_gentity.h>
#include <gdl/gdl_locus_type.h>
#include <gdl/gdl_gview.h>
#include <gdl/gdl_gmap.h>

__BEGIN_DECLS

typedef struct _gdl_dna_reader gdl_dna_reader;

typedef struct
{
	size_t npoly;
	size_t * poly;
	gdl_boolean indel;
	size_t      from;
	size_t      to;
	gdl_locus  * locus;
	gdl_string * buffer;
} gdl_dna_locus;

gdl_dna_locus * gdl_dna_locus_alloc (size_t position, const char base);
void gdl_dna_locus_free (gdl_dna_locus * d);
void gdl_dna_locus_poly (gdl_dna_locus * d, const size_t * poly);

typedef struct 
{
	gdl_string * name;
	gdl_string * acronym;
	int (*read)(gdl_dna_reader * r);
} gdl_dna_reader_type;

struct _gdl_dna_reader
{
	const gdl_dna_reader_type * type;
	const gdl_string * base;
	gdl_string * filename;
	gdl_string * error;
	FILE       * stream;
	gdl_gview * gview;
	gdl_gmap  * gmap;
	gdl_locus_type_registry * ltype;
	gdl_boolean remove_indel;
	gdl_boolean collapse_indel;
	gdl_boolean reverse;
};

gdl_dna_reader * gdl_dna_reader_alloc (const gdl_dna_reader_type * type, const gdl_string * base, gdl_boolean remove_indel, gdl_boolean collapse_indel, gdl_boolean reverse);
void gdl_dna_reader_free (gdl_dna_reader * reader);
int gdl_dna_reader_open (gdl_dna_reader * reader, const gdl_string * filename);
int gdl_dna_reader_parse (gdl_dna_reader * reader);
gdl_gview * gdl_dna_reader_get_data (const gdl_dna_reader * reader);
gdl_gmap  * gdl_dna_reader_get_map (const gdl_dna_reader * reader);
gdl_locus_type_registry * gdl_dna_reader_get_locus_type (const gdl_dna_reader * reader);
int gdl_dna_reader_close (gdl_dna_reader * reader);
int gdl_dna_reader_error (gdl_dna_reader * reader, char * format, ...);

gdl_boolean gdl_dna_reader_is_valid (const int base);
gdl_boolean gdl_dna_reader_is_ambiguous (const int base);
gdl_boolean gdl_dna_reader_has_ambiguous (const gdl_string * seq);
gdl_boolean gdl_dna_reader_is_gap (const int base);
void gdl_dna_reader_collapse_indel (const gdl_string * refseq, const size_t * poly, gdl_dna_locus ** sites, size_t * nl);
gdl_gmap * gdl_dna_reader_create_map (gdl_dna_locus ** sites, size_t nl, size_t length, gdl_boolean reverse);
gdl_locus_type_registry * gdl_dna_reader_create_locus_type (gdl_dna_locus ** sites, size_t nl);

GDL_VAR const gdl_dna_reader_type * gdl_dna_reader_fasta;

__END_DECLS

#endif /* __GDL_DNA_READER_H__ */
