/*
 *  eqtl/bpermut.c 
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:47 $, $Version$
 *  
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */

#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_eqtl_bayes.h>

gdl_eqtl_bayes_permut *
gdl_eqtl_bayes_permut_alloc (const size_t nf_bin, const size_t nbf_bin, const size_t npv_bin, const double log_min, const double log_max)
{
	size_t i,j;
	gdl_eqtl_bayes_permut * p;
	
	p = GDL_MALLOC (gdl_eqtl_bayes_permut, 1);
	
	p->nf_bin  = nf_bin;
	p->nbf_bin = nbf_bin;
	p->npv_bin = npv_bin;
	p->log_min = log_min;
	p->log_max = log_max;
	p->counts  = GDL_MALLOC (size_t **, p->nf_bin);
	for(i = 0; i < p->nf_bin; i++)
	{
		p->counts[i] = GDL_MALLOC (size_t *, p->nbf_bin);
		for(j = 0; j < p->nbf_bin; j++)
		{
			p->counts[i][j] = GDL_CALLOC (size_t, p->npv_bin);
		}
	}
	return p;	
}

void
gdl_eqtl_bayes_permut_free (gdl_eqtl_bayes_permut * p)
{
	if (p)
	{
		size_t i,j;
		for(i = 0; i < p->nf_bin; i++)
		{
			for(j = 0; j < p->nbf_bin; j++)
				GDL_FREE (p->counts[i][j]);
			GDL_FREE (p->counts[i]);
		}
		GDL_FREE (p->counts);
		GDL_FREE (p);
	}	
}

static int
double2bin (const gdl_eqtl_bayes_permut * p, const double x, const char type)
{
	int bin;
	double y;
	
	switch(type)
	{
		case 'f': // MAF
			y = (x > 0.5) ? 1-x : x;
			bin = floor(2*y*p->nf_bin);
			return (bin < p->nf_bin) ? bin : p->nf_bin-1;
		case 'b': // BF
		   y = (x-p->log_min)/(p->log_max-p->log_min);
			if (y > 1) y = 1;
			else if (y < 0) y = 0;
			bin = floor(y*p->nbf_bin);
			return (bin < p->nbf_bin) ? bin : p->nbf_bin-1;
		case 'p': // P-value	
			y = (x-p->log_min)/(p->log_max-p->log_min);
			if (y > 1) y = 1;
			else if (y < 0) y = 0;
			bin = floor(y*p->npv_bin);
			return (bin < p->npv_bin) ? bin : p->npv_bin-1;
		default:
			return -1;
	}
} 

static double
bin2double (const gdl_eqtl_bayes_permut * p, const size_t bin, const char type)
{
	double y;
	
	switch(type)
	{
		case 'f': // MAF
			return 0.5*((double)(bin))/p->nf_bin;
		case 'b': // BF
			return p->log_min + (p->log_max-p->log_min)*(((double)(bin))/p->nbf_bin);
		case 'p': // P-value	
			return p->log_min + (p->log_max-p->log_min)*(((double)(bin))/p->npv_bin);
		default:
			return 0;
	}
}

void
gdl_eqtl_bayes_permut_add (gdl_eqtl_bayes_permut * p, const double snp_maf, const double snp_logbf, const double snp_logpv)
{
	int i,j,k;
	
	i = double2bin (p,snp_maf,'f');
	j = double2bin (p,snp_logbf,'b');
	k = double2bin (p,snp_logpv,'p');
	
	if ((i==-1 || j==-1) || k==-1) return;
	
	//printf ("ADD (%g,%g,%g) ==> (%d,%d,%d) ==> (%g,%g,%g)\n", snp_maf, snp_logbf, snp_logpv, i, j, k, bin2double(p,i,'f'), bin2double(p,j,'b'), bin2double(p,k,'p'));
	(p->counts[i][j][k])++;
}

int
gdl_eqtl_bayes_permut_fprintf (FILE * stream, const gdl_eqtl_bayes_permut * p)
{
	if (p)
	{
		//fprintf (stream, "MAF BF PV N\n");
		size_t i,j,k;
		for(i = 0; i < p->nf_bin; i++)
			for(j = 0; j < p->nbf_bin; j++)
				for(k = 0; k < p->npv_bin; k++)
					fprintf (stream, "%g %g %g %d\n", bin2double(p,i,'f'), bin2double(p,j,'b'), bin2double(p,k,'p'), p->counts[i][j][k]);
	}	
}
