/*
 *  eqtl/chi.c 
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:47 $, $Version$
 *  
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */
#include <math.h>

#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_string.h>
#include <gdl/gdl_matrix.h>
#include <gdl/gdl_vector.h>
#include <gdl/gdl_rng.h>
#include <gdl/gdl_randist.h>
#include <gdl/gdl_specfunc.h>
#include <gdl/gdl_statistics_double.h>
#include <gdl/gdl_snp_annotation.h>
#include <gdl/gdl_eqtl_chromosome.h>

//void gdl_eqtl_chromosome_cis_get_snp_vector (gdl_eqtl_chromosome * chrom, const size_t snp, double * x, double * pop_freq, double * freq);
void gdl_eqtl_chromosome_get_snp_vector (gdl_eqtl_chromosome * chrom, const size_t snp, double * x, double * pop_freq, double * freq);
double ** gdl_eqtl_chromosome_cis_get_block_signal (gdl_eqtl_chromosome * chrom, gdl_eqtl_block * block, double ** var, size_t ** pidx, size_t * np);
double gdl_eqtl_chromosome_do_LRT (gdl_eqtl_chromosome * chrom, const double * x, const double * y, const double * popf, double * a, double * r2, double * p);

static void 
_cis_print_annot (FILE * stream, const gdl_snp_annot_dico * dico, const gdl_snp * snp, const gdl_string * gene)
{
	if (snp->annot)
	{
		if (snp->annot->geno) fprintf (stream, "\tMC");
		else fprintf (stream, " -");
		if (snp->annot->gene && !strcmp(gene, dico->gene[snp->annot->gene->gene_id]->name))
		   fprintf (stream, " %s", dico->func[snp->annot->gene->func_id]->name);
		else fprintf(stream, " -");
	}
	else fprintf (stream, " - -");
}

void
gdl_eqtl_chromosome_cis_test (gdl_eqtl_chromosome * chrom, gdl_snp_annot_dico * dico, const double freqmin, const double pmin, gdl_boolean best_pval)
{
	const size_t G = chrom->ngene;
	const size_t P = chrom->ploidy;
	const size_t N = chrom->nindiv;
	const size_t POP = chrom->npop;
	size_t i, j, k, l, p, s, * pidx, np, * minl;
	gdl_genex_gene * gene;
	gdl_genex_block * block;
	gdl_snp ** snps;
	double * popf, f, chi, t, pv, r2, * x, ** Y, * var;
	double * minpv;
	
	snps = chrom->snp_data->chrom->snps;
	
	x = GDL_MALLOC (double, N);
	popf = GDL_MALLOC (double, POP);
	
	for (i = 0; i < G; i++)
	{
		gene = chrom->genes[i];
		
		if (gene->ignore == 'y') continue;
		
		for (j = 0; j < gene->size; j++)
		{
			block = gene->blocks[j];
			
			if (block->ignore == 'y') continue;
			
			gdl_genex_block_clean_record_snp (block);
			
			Y = gdl_eqtl_chromosome_cis_get_block_signal (chrom, block, &var, &pidx, &np);
			
			if (best_pval)
			{
				minpv  = GDL_CALLOC (double, np);
				minl   = GDL_CALLOC (size_t, np);
			}
			
			for (l = block->snp_up; l <= block->snp_down; l++)
			{
				if (snps[l]->ignore == 'y') continue;
				
				gdl_eqtl_chromosome_get_snp_vector (chrom, l, x, popf, &f);
				
				for (p = 0; p < np; p++)
				{
					chi = gdl_eqtl_chromosome_do_LRT (chrom, x, Y[p], popf, &t, &r2, &pv);
					
					if (!best_pval && pv < pmin)
					{
						fprintf (chrom->logger, "%s %s %d %s %d %d %d %d %c %s %d %d %1.5f",chrom->name, snps[l]->rs, snps[l]->position, gene->name,gene->id,j+1,block->start,block->end,block->strand,block->probes[pidx[p]]->name, block->probes[pidx[p]]->start, block->probes[pidx[p]]->end, f/(double)P);
						for (s = 0; s < chrom->npop; s++)
						{
							fprintf (chrom->logger, " %1.5f", popf[s]/(double)P);
						}
						fprintf (chrom->logger, " %g %g %g %e\n", chi, t, r2, pv);
						gdl_genex_block_record_snp (block, pidx[p], l);
						//gdl_genex_annot_dico_add_count (block, snps[l], dico);
					}
					else if (best_pval && (pv < minpv[p] || !minpv[p]))
					{
						minpv[p] = pv;
						minl[p]  = l;
					}
				}
			}
			if (best_pval)
			{
				for (p = 0; p < np; p++)
				{
					if (minpv[p] < pmin)
					{
						gdl_eqtl_chromosome_cis_get_snp_vector (chrom, minl[p], x, popf, &f);
						chi = gdl_eqtl_chromosome_do_LRT (chrom, x, Y[p], popf, &t, &r2, &pv);
						fprintf (chrom->logger, "%s %s %d %s %d %d %d %d %c %s %d %d %1.5f",chrom->name, snps[l]->rs, snps[l]->position, gene->name,gene->id,j+1,block->start,block->end,block->strand,block->probes[pidx[p]]->name, block->probes[pidx[p]]->start, block->probes[pidx[p]]->end, f/(double)P);
						for (s = 0; s < chrom->npop; s++)
						{
							fprintf (chrom->logger, " %1.5f", popf[s]/(double)P);
						}
						fprintf (chrom->logger, " %g %g %g %e\n", chi, t, r2, pv);
						gdl_genex_block_record_snp (block, pidx[p], l);
						//gdl_genex_annot_dico_add_count (block, snps[minl[p]], dico);
					}
				}
				GDL_FREE (minpv);	
				GDL_FREE (minl);
			}
			GDL_MATRIX_FREE (Y, np);
			GDL_FREE (var);
			GDL_FREE (pidx);
		}
	}
}
