/*
 *  eqtl/cnv.c 
 * 
 *  $Author: baptiste $, $Date: 2008-05-13 15:33:47 $, $Version$
 *  
 *  Libgdl : a C library for statistical genetics
 * 
 *  Copyright (C) 2003-2006  Jean-Baptiste Veyrieras, INRA, France.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA * 
 */
#include <math.h>

#include <gdl/gdl_common.h>
#include <gdl/gdl_errno.h>
#include <gdl/gdl_string.h>
#include <gdl/gdl_matrix.h>
#include <gdl/gdl_vector.h>
#include <gdl/gdl_rng.h>
#include <gdl/gdl_randist.h>
#include <gdl/gdl_specfunc.h>
#include <gdl/gdl_statistics_double.h>
#include <gdl/gdl_snp_annotation.h>
#include <gdl/gdl_eqtl_chromosome.h>

double ** gdl_eqtl_chromosome_cis_get_block_signal (gdl_eqtl_chromosome * chrom, gdl_eqtl_block * block, double ** var, size_t ** pidx, size_t * np);

void
gdl_eqtl_chromosome_cis_get_cnv_vector (gdl_eqtl_chromosome * chrom, const size_t cnv, double x[], double pop_mean[], size_t pop_size[])
{
	size_t i, j, k, p;
	for (k = i = 0; i < chrom->npop; i++)
	{
		gdl_cnv_data * data = chrom->snp_data->cnvs[i];
		pop_mean[i] = 0;
		pop_size[i] = 0;
		for(j = 0; j < chrom->pop_sizes[i]; j++, k++)
		{
			x[k] = gdl_cnv_data_get (data, j, cnv);
			if (!gdl_isnan (x[k]))
			{
				pop_mean[i]+=x[k];
				pop_size[i]++;
			}
		}
		pop_mean[i] /= pop_size[i];
	}
}

double ** 
gdl_eqtl_chromosome_cis_cnv_block_signal (gdl_eqtl_chromosome * chrom, gdl_eqtl_block * block, size_t ** pidx, size_t * np)
{
	size_t i, j, k, p, pp;
	double ** Y, s, m;
	
	for (*np = p = 0; p < block->size; p++)
	{
	   if (block->probes[p]->ignore=='n') (*np)++;
	}
	
	Y     = GDL_MATRIX_ALLOC (double, *np, chrom->nindiv+1);
	*pidx = GDL_MALLOC (size_t, *np);
	
	for (pp = p = 0; p < block->size; p++)
	{
		if (block->probes[p]->ignore == 'y') continue;
		(*pidx)[pp] = p;
		for (k = i = 0; i < chrom->npop; i++)
		{
			for(j = 0; j < chrom->pop_sizes[i]; j++, k++)
			{
				Y[pp][k] = block->probes[p]->data[i][j];
			}
		}
		pp++;
	}
	
	return Y;
}

double
gdl_eqtl_chromosome_cis_cnv_LRT (gdl_eqtl_chromosome * chrom, const double x[], const double y[], const double xpop_mean[], double ypop_mean[], const size_t pop_size[], double * a, double * r2, double * p)
{
	size_t i, j, k;
	double l0, l1, chi, e, sig0=0, sig1=0, xx=0, xy=0;
	
	for (k = i = 0; i < chrom->npop; i++)
	{
		for(j = 0; j < chrom->pop_sizes[i]; j++, k++)
		{
			if (!gdl_isnan (x[k]))
			{
				ypop_mean[i] += y[k];
			}
		}
		ypop_mean[i] /= pop_size[i];
	}
	
	for (k = i = 0; i < chrom->npop; i++)
	{
		for(j = 0; j < chrom->pop_sizes[i]; j++, k++)
		{
			if (!gdl_isnan (x[k]))
			{
				e = x[k]-xpop_mean[i];
				xx += e*e;
				xy += e*(y[k]-ypop_mean[i]);
			}
		}
	}
	
	*a=xy/xx;
	
	for (k = i = 0; i < chrom->npop; i++)
	{
		for(j = 0; j < chrom->pop_sizes[i]; j++, k++)
		{
			if (!gdl_isnan (x[k]))
			{
			   e =  (y[k]-ypop_mean[i]);
			   sig0 += e*e;
			   e -= (*a)*(x[k]-xpop_mean[i]);
			   sig1 += e*e;
			}
		}
	}
	chi = (log(sig0)-log(sig1))*k;
	chi = (chi >= 0) ? chi : 0;
	*p=gdl_sf_gamma_inc_Q (0.5,0.5*chi);
	*r2=1-sig1/sig0;
	return chi;
}

void
gdl_eqtl_chromosome_cnv_cis_test (gdl_eqtl_chromosome * chrom)
{
	const size_t G = chrom->ngene;
	const size_t P = chrom->ploidy;
	const size_t N = chrom->nindiv;
	const size_t POP = chrom->npop;
	size_t i, j, k, l, p, * pidx, np, * minl, NC, * pop_size;
	gdl_genex_gene * gene;
	gdl_genex_block * block;
	gdl_snp ** snps;
	double * ypop_mean, * xpop_mean, f, chi, t, pv, r2, * x, ** Y, * var;
	
	x = GDL_MALLOC (double, N);
	xpop_mean = GDL_MALLOC (double, POP);
	ypop_mean = GDL_MALLOC (double, POP);
	pop_size = GDL_MALLOC (size_t, POP);
	
	for (i = 0; i < G; i++)
	{
		gene = chrom->genes[i];
		
		if (gene->ignore == 'y') continue;
		
		for (j = 0; j < gene->size; j++)
		{
			block = gene->blocks[j];
			
			if (block->ignore == 'y') continue;
			
			Y = gdl_eqtl_chromosome_cis_cnv_block_signal (chrom, block, &pidx, &np);
			
			gdl_cnv_clone ** clones = gdl_snp_chromosome_get_cnv_clones (chrom->snp_data, snps[block->snp_up]->position, snps[block->snp_down]->position, &NC);
			
			for (l = 0; l <= NC; l++)
			{
				gdl_eqtl_chromosome_cis_get_cnv_vector (chrom, clones[l]->idx, x, xpop_mean, pop_size);
				
				for (p = 0; p < np; p++)
				{
					
					chi = gdl_eqtl_chromosome_cis_cnv_LRT (chrom, x, Y[p], xpop_mean, ypop_mean, pop_size, &t, &r2, &pv);
					fprintf (chrom->logger, "%s %s %d %d %c %s %d %d %s %d %d %g %g %g %e",
					                         chrom->name,
					                         gene->name,
					                         block->start,
					                         block->end,
					                         block->strand,
					                         block->probes[pidx[p]]->name,
					                         block->probes[pidx[p]]->start,
					                         block->probes[pidx[p]]->end,
					                         clones[l]->name,
					                         clones[l]->start,
					                         clones[l]->end, chi, t, r2, pv);
				}
			}
			
			GDL_MATRIX_FREE (Y, np);
			GDL_FREE (var);
			GDL_FREE (pidx);
		}
	}
	
	GDL_FREE (x);
	GDL_FREE (ypop_mean);
	GDL_FREE (xpop_mean);
	GDL_FREE (pop_size);
}
